// Sample data - in a real app, you would fetch this from an API
        const stockData = {
            daily: {
                labels: ['Jan 1', 'Jan 2', 'Jan 3', 'Jan 4', 'Jan 5', 'Jan 6', 'Jan 7'],
                values: [150, 152, 148, 155, 153, 157, 160],
                volume: [10000, 12000, 8000, 15000, 11000, 13000, 14000]
            },
            monthly: {
                labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul'],
                values: [150, 155, 160, 158, 165, 170, 175],
                volume: [300000, 320000, 310000, 290000, 330000, 340000, 350000]
            },
            quarterly: {
                labels: ['Q1', 'Q2', 'Q3', 'Q4'],
                values: [155, 165, 172, 180],
                volume: [950000, 980000, 1000000, 1050000]
            },
            yearly: {
                labels: ['2020', '2021', '2022', '2023', '2024'],
                values: [120, 140, 150, 165, 180],
                volume: [3800000, 4000000, 4200000, 4400000, 4600000]
            }
        };

        const distributionData = {
            labels: ['Technology', 'Healthcare', 'Financial', 'Consumer', 'Industrial', 'Energy'],
            values: [35, 20, 15, 12, 10, 8],
            colors: [
                'rgba(54, 162, 235, 0.7)',
                'rgba(75, 192, 192, 0.7)',
                'rgba(255, 206, 86, 0.7)',
                'rgba(153, 102, 255, 0.7)',
                'rgba(255, 159, 64, 0.7)',
                'rgba(255, 99, 132, 0.7)'
            ]
        };

        // Initialize charts
        let movementChart;
        let distributionChart;

        function initCharts() {
            const movementCtx = document.getElementById('movementChart').getContext('2d');
            const distributionCtx = document.getElementById('distributionChart').getContext('2d');
            
            // Movement Chart (Bar)
            movementChart = new Chart(movementCtx, {
                type: 'bar',
                data: {
                    labels: stockData.daily.labels,
                    datasets: [{
                        label: 'Stock Price',
                        data: stockData.daily.values,
                        backgroundColor: 'rgba(54, 162, 235, 0.7)',
                        borderColor: 'rgba(54, 162, 235, 1)',
                        borderWidth: 1,
                        yAxisID: 'y'
                    }, {
                        label: 'Volume',
                        data: stockData.daily.volume,
                        backgroundColor: 'rgba(201, 203, 207, 0.5)',
                        borderColor: 'rgba(201, 203, 207, 1)',
                        borderWidth: 1,
                        type: 'line',
                        yAxisID: 'y1'
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        y: {
                            type: 'linear',
                            display: true,
                            position: 'left',
                            title: {
                                display: true,
                                text: 'Price ($)'
                            }
                        },
                        y1: {
                            type: 'linear',
                            display: true,
                            position: 'right',
                            grid: {
                                drawOnChartArea: false
                            },
                            title: {
                                display: true,
                                text: 'Volume'
                            }
                        },
                        x: {
                            title: {
                                display: true,
                                text: 'Date'
                            }
                        }
                    },
                    plugins: {
                        title: {
                            display: true,
                            text: 'Stock Price Movement'
                        },
                        tooltip: {
                            mode: 'index',
                            intersect: false
                        }
                    }
                }
            });
            
            // Distribution Chart (Pie)
            distributionChart = new Chart(distributionCtx, {
                type: 'pie',
                data: {
                    labels: distributionData.labels,
                    datasets: [{
                        data: distributionData.values,
                        backgroundColor: distributionData.colors,
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        title: {
                            display: true,
                            text: 'Stock Distribution by Sector'
                        },
                        legend: {
                            position: 'right'
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    const label = context.label || '';
                                    const value = context.raw || 0;
                                    const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                    const percentage = Math.round((value / total) * 100);
                                    return `${label}: ${value} (${percentage}%)`;
                                }
                            }
                        }
                    }
                }
            });
        }

        // Filter data based on time period
        function filterData(period) {
            // Update active button
            document.querySelectorAll('.filters button').forEach(btn => {
                btn.classList.remove('active');
                if (btn.textContent.toLowerCase() === period) {
                    btn.classList.add('active');
                }
            });
            
            // Update chart data
            movementChart.data.labels = stockData[period].labels;
            movementChart.data.datasets[0].data = stockData[period].values;
            movementChart.data.datasets[1].data = stockData[period].volume;
            movementChart.update();
            
            // Update summary
            const summaryEl = document.getElementById('summary');
            const avgValue = (stockData[period].values.reduce((a, b) => a + b, 0) / stockData[period].values.length).toFixed(2);
            const totalVolume = stockData[period].volume.reduce((a, b) => a + b, 0).toLocaleString();
            
            summaryEl.innerHTML = `
                <h3>${period.charAt(0).toUpperCase() + period.slice(1)} Summary</h3>
                <p><strong>Average Price:</strong> $${avgValue}</p>
                <p><strong>Total Volume:</strong> ${totalVolume}</p>
                <p><strong>Price Change:</strong> $${stockData[period].values[0]} to $${stockData[period].values[stockData[period].values.length - 1]}</p>
                <p><strong>Percentage Change:</strong> ${(((stockData[period].values[stockData[period].values.length - 1] - stockData[period].values[0]) / stockData[period].values[0]) * 100).toFixed(2)}%</p>
            `;
        }

        // Initialize the dashboard when the page loads
        window.onload = function() {
            initCharts();
            filterData('daily');
        };