/**
 * API Configuration
 *
 * This file centralizes all API endpoints to prevent hardcoded URLs.
 * All API calls should use these endpoints instead of hardcoded URLs.
 */
window.AppConfig = {
    // Base URL - empty for relative paths (recommended)
    baseUrl: '',

    // API endpoints - use relative paths to work with any domain
    api: {
        // Customer endpoints
        customers: '/api/v1/customers',
        customer: (id) => `/api/v1/customers/${id}`,

        // RFQ endpoints
        rfqs: '/api/v1/rfqs',
        rfq: (id) => `/api/v1/rfqs/${id}`,
        rfqDetails: (id) => `/admin/rfq-details/${id}`,

        // Quotation endpoints
        quotations: '/api/v1/quotations',
        quotation: (id) => `/api/v1/quotations/${id}`,

        // Raw Materials endpoints
        rawMaterials: '/api/v1/raw-materials',
        rawMaterial: (id) => `/api/v1/raw-materials/${id}`,

        // Sales Order endpoints
        salesOrders: '/api/v1/sales-orders',
        salesOrder: (id) => `/api/v1/sales-orders/${id}`,

        // Invoice endpoints
        invoices: '/api/v1/invoices',
        invoice: (id) => `/api/v1/invoices/${id}`,
    },

    /**
     * Build full URL for an endpoint
     * @param {string} endpoint - The API endpoint path
     * @returns {string} - Full URL
     */
    getUrl: function(endpoint) {
        return this.baseUrl + endpoint;
    },

    /**
     * Get CSRF token from meta tag
     * @returns {string} - CSRF token
     */
    getCsrfToken: function() {
        const token = document.querySelector('meta[name="csrf-token"]');
        return token ? token.getAttribute('content') : '';
    },

    /**
     * Default fetch options with CSRF token
     * @param {object} options - Additional fetch options
     * @returns {object} - Merged fetch options
     */
    fetchOptions: function(options = {}) {
        const defaults = {
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': this.getCsrfToken(),
                'Accept': 'application/json',
            },
        };

        return {
            ...defaults,
            ...options,
            headers: {
                ...defaults.headers,
                ...(options.headers || {}),
            },
        };
    },
};

// Export for ES modules if supported
if (typeof module !== 'undefined' && module.exports) {
    module.exports = window.AppConfig;
}
