// Sample data - in a real app, you would fetch this from an API
        const accountingData = {
            daily: {
                labels: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
                revenue: [1200, 1900, 1700, 2100, 2300, 1500, 1000],
                expenses: [800, 1100, 900, 1200, 1500, 800, 600]
            },
            monthly: {
                labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'],
                revenue: [45000, 42000, 48000, 51000, 55000, 58000, 60000, 62000, 59000, 63000, 65000, 70000],
                expenses: [38000, 35000, 40000, 42000, 45000, 47000, 49000, 50000, 48000, 51000, 53000, 55000]
            },
            quarterly: {
                labels: ['Q1', 'Q2', 'Q3', 'Q4'],
                revenue: [135000, 164000, 181000, 198000],
                expenses: [113000, 134000, 147000, 159000]
            },
            yearly: {
                labels: ['2020', '2021', '2022', '2023', '2024'],
                revenue: [520000, 580000, 640000, 710000, 780000],
                expenses: [450000, 490000, 530000, 580000, 620000]
            }
        };

        const costData = {
            labels: ['Salaries', 'Rent', 'Utilities', 'Marketing', 'Supplies', 'Other'],
            values: [45, 20, 10, 15, 5, 5],
            colors: [
                'rgba(54, 162, 235, 0.7)',
                'rgba(255, 99, 132, 0.7)',
                'rgba(75, 192, 192, 0.7)',
                'rgba(255, 206, 86, 0.7)',
                'rgba(153, 102, 255, 0.7)',
                'rgba(255, 159, 64, 0.7)'
            ]
        };

        const pendingInvoices = [
            { id: 'INV-2023-001', date: '2023-10-15', description: 'Web Design Services', amount: 2500, dueDate: '2023-11-15', status: 'pending' },
            { id: 'INV-2023-002', date: '2023-10-20', description: 'Monthly Maintenance', amount: 1200, dueDate: '2023-11-20', status: 'pending' },
            { id: 'INV-2023-003', date: '2023-09-25', description: 'Consulting Fees', amount: 3500, dueDate: '2023-10-25', status: 'overdue' },
            { id: 'INV-2023-004', date: '2023-11-01', description: 'Software License', amount: 800, dueDate: '2023-12-01', status: 'pending' },
            { id: 'INV-2023-005', date: '2023-10-10', description: 'Hosting Services', amount: 450, dueDate: '2023-11-10', status: 'paid' }
        ];

        // Initialize charts
        let accountingChart;
        let costChart;

        function initCharts() {
            const accountingCtx = document.getElementById('accountingChart').getContext('2d');
            const costCtx = document.getElementById('costChart').getContext('2d');
            
            // Accounting Chart (Bar)
            accountingChart = new Chart(accountingCtx, {
                type: 'bar',
                data: {
                    labels: accountingData.daily.labels,
                    datasets: [
                        {
                            label: 'Revenue',
                            data: accountingData.daily.revenue,
                            backgroundColor: 'rgba(46, 204, 113, 0.7)',
                            borderColor: 'rgba(46, 204, 113, 1)',
                            borderWidth: 1
                        },
                        {
                            label: 'Expenses',
                            data: accountingData.daily.expenses,
                            backgroundColor: 'rgba(231, 76, 60, 0.7)',
                            borderColor: 'rgba(231, 76, 60, 1)',
                            borderWidth: 1
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        y: {
                            beginAtZero: true,
                            title: {
                                display: true,
                                text: 'Amount ($)'
                            }
                        },
                        x: {
                            title: {
                                display: true,
                                text: 'Period'
                            }
                        }
                    },
                    plugins: {
                        title: {
                            display: true,
                            text: 'Revenue vs Expenses'
                        },
                        tooltip: {
                            mode: 'index',
                            intersect: false
                        }
                    }
                }
            });
            
            // Cost Chart (Pie)
            costChart = new Chart(costCtx, {
                type: 'pie',
                data: {
                    labels: costData.labels,
                    datasets: [{
                        data: costData.values,
                        backgroundColor: costData.colors,
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        title: {
                            display: true,
                            text: 'Expense Breakdown'
                        },
                        legend: {
                            position: 'right'
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    const label = context.label || '';
                                    const value = context.raw || 0;
                                    return `${label}: ${value}%`;
                                }
                            }
                        }
                    }
                }
            });
        }

        // Filter data based on time period
        function filterData(period) {
            // Update active button
            document.querySelectorAll('.filters button').forEach(btn => {
                btn.classList.remove('active');
                if (btn.textContent.toLowerCase() === period) {
                    btn.classList.add('active');
                }
            });
            
            // Update chart data
            accountingChart.data.labels = accountingData[period].labels;
            accountingChart.data.datasets[0].data = accountingData[period].revenue;
            accountingChart.data.datasets[1].data = accountingData[period].expenses;
            accountingChart.update();
            
            // Update summary
            updateSummary(period);
        }

        // Update financial summary
        function updateSummary(period) {
            const summaryEl = document.getElementById('summary');
            const data = accountingData[period];
            
            const totalRevenue = data.revenue.reduce((a, b) => a + b, 0);
            const totalExpenses = data.expenses.reduce((a, b) => a + b, 0);
            const profitLoss = totalRevenue - totalExpenses;
            const profitMargin = ((profitLoss / totalRevenue) * 100).toFixed(2);
            
            summaryEl.innerHTML = `
                <div class="summary-item">
                    <h4>Total Revenue</h4>
                    <p>$${totalRevenue.toLocaleString()}</p>
                </div>
                <div class="summary-item">
                    <h4>Total Expenses</h4>
                    <p>$${totalExpenses.toLocaleString()}</p>
                </div>
                <div class="summary-item">
                    <h4>Net ${profitLoss >= 0 ? 'Profit' : 'Loss'}</h4>
                    <p class="${profitLoss >= 0 ? 'profit' : 'loss'}">$${Math.abs(profitLoss).toLocaleString()}</p>
                </div>
                <div class="summary-item">
                    <h4>Profit Margin</h4>
                    <p class="${profitMargin >= 0 ? 'profit' : 'loss'}">${profitMargin}%</p>
                </div>
            `;
        }

        // Populate invoices table
        function populateInvoicesTable() {
            const tableBody = document.querySelector('#invoicesTable tbody');
            tableBody.innerHTML = '';
            
            pendingInvoices.forEach(invoice => {
                const row = document.createElement('tr');
                const today = new Date();
                const dueDate = new Date(invoice.dueDate);
                let status = invoice.status;
                
                // Automatically mark as overdue if due date has passed
                if (status === 'pending' && dueDate < today) {
                    status = 'overdue';
                }
                
                row.innerHTML = `
                    <td>${invoice.id}</td>
                    <td>${invoice.date}</td>
                    <td>${invoice.description}</td>
                    <td>$${invoice.amount.toLocaleString()}</td>
                    <td>${invoice.dueDate}</td>
                    <td class="status-${status}">${status.charAt(0).toUpperCase() + status.slice(1)}</td>
                `;
                tableBody.appendChild(row);
            });
        }

        // Initialize the dashboard when the page loads
        window.onload = function() {
            initCharts();
            filterData('daily');
            populateInvoicesTable();
        };