/**
 * ----------------------------- script for edit button-------------------------------------------------------------------
 */

// Use event delegation for the edit buttons
document.getElementById('dataTable').addEventListener('click', function(event) {
    if (event.target && event.target.classList.contains('edit-btn')) {
        const customerId = event.target.closest('button').dataset.id; // Assuming first cell has the customer ID

        // Store the customer ID in session storage
        sessionStorage.setItem('currentCustId', customerId);

        // Redirect to the form page for editing details
        window.location.href = 'customer-details.html';
    }
});

/**
 * ----------------------------- script for import details-------------------------------------------------------------------
 */

// Trigger the modal when the Import button is clicked
document.getElementById('import-button').addEventListener('click', function() {
    $('#importModal').modal('show');
});

// Handle the Excel file import
document.getElementById('importDataBtn').addEventListener('click', async function () {
    let fileInput = document.getElementById('excelFile');
    let file = fileInput.files[0]; // Get the uploaded file

    if (file) {
        let reader = new FileReader();
        
        reader.onload = async function (e) {
            let data = e.target.result;
            let workbook = XLSX.read(data, { type: 'binary' });
            
            // Assuming the first sheet contains the data
            let sheetName = workbook.SheetNames[0];
            let sheet = workbook.Sheets[sheetName];
            
            // Convert sheet to JSON format
            let excelData = XLSX.utils.sheet_to_json(sheet);

            // Fetch existing companies from the API
            let existingCompanies = [];
            try {
                const response = await fetch('/api/v1/customers');
                if (!response.ok) {
                    throw new Error(`Failed to fetch existing customers: ${response.statusText}`);
                }
                existingCompanies = await response.json();
            } catch (error) {
                console.error("Error fetching existing customer data:", error);
            }

            for (const row of excelData) {
                const companyName = row['Company Name']?.trim();
                if (!companyName) continue; // Skip if company name is empty

                // Check if company already exists
                let existingCompany = existingCompanies.find(c => c.companyName.trim() === companyName);

                let companyId;
                if (existingCompany) {
                    // Prompt user for confirmation to overwrite
                    // const overwrite = confirm(`Company "${companyName}" already exists. Do you want to overwrite it?`);
                    // if (!overwrite) continue; // Skip updating if user cancels

                    companyId = existingCompany.companyId1; // Keep the existing ID
                } else {
                    // Generate a new company ID
                    const rolePrefix = getRolePrefix(row['Company Role']);
                    companyId = await generateCompanyId(rolePrefix);
                }

                // Generate the company logo (acronym or initials)
                const companyLogo1 = generateCompanyLogo(row['Company Acronym'], row['Company Name']);

                // Collect the Contact Person details
                let contactPersons = [];
                if (row['Contact Person Name']) {
                    const contactPersonNames = row['Contact Person Name'].split('<br>').map(name => name.trim());
                    const contactPersonEmails = row['Contact Person Email']?.split('<br>').map(email => email.trim());
                    const contactPersonMobiles = row['Contact Person Mobile']?.split('<br>').map(mobile => mobile.trim());
                    const contactPersonDesignations = row['Contact Person Designation']?.split('<br>').map(designation => designation.trim());

                    // Add each contact person to the array
                    for (let i = 0; i < contactPersonNames.length; i++) {
                        contactPersons.push({
                            name: contactPersonNames[i] || '',
                            email: contactPersonEmails[i] || '',
                            mobile: contactPersonMobiles[i] || '',
                            designation: contactPersonDesignations[i] || ''
                        });
                    }
                }

                // Prepare company data
                const companyData = {
                    companyId1: companyId,
                    companyLogo: companyLogo1 || '',
                    companyName: row['Company Name'] || '',
                    acronym: row['Company Acronym'] || '',
                    companyEmail: row['Company Email'] || '',
                    companyContact: row['Primary Contact Number'] || '',
                    secondaryContact: row['Secondary Contact Number'] || '',
                    companyType: row['Company Type'] || '',
                    gstNo: row['GST No.'] || '',
                    panNo: row['PAN No.'] || '',
                    msmeNo: row['MSME/Udyam No.'] || '',
                    iecNo: row['IEC No.'] || '',
                    websiteUrl: row['Website'] || '',
                    billingAddress: row['Billing Address'] || '',
                    shippingAddress: row['Shipping Address'] || '',
                    bankName: row['Bank Name'] || '',
                    branchName: row['Bank Branch Name'] || '',
                    accountNumber: row['Account Number'] || '',
                    accountType: row['Account Type'] || '',
                    accountHolderName: row['Account Holder Name'] || '',
                    accountCurrency: row['Account Currency'] || '',
                    ifscCode: row['IFSC Code'] || '',
                    swiftCode: row['SWIFT/BIC Code'] || '',
                    ContactPerson: contactPersons 
                };

                // If company exists, update it; otherwise, create a new entry
                try {
                    let apiUrl = existingCompany
                        ? `/api/v1/customers${existingCompany.id}/` // Update API URL
                        : '/api/v1/customers'; // Create new

                    const method = existingCompany ? 'PUT' : 'POST'; // Use PUT for update, POST for new

                    const response = await fetch(apiUrl, {
                        method: method,
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify(companyData),
                    });

                    // const responseText = await response.text(); // Get the raw response

                    if (!response.ok) {
                        throw new Error(`Failed to ${existingCompany ? 'update' : 'create'} company: ${response.statusText}`);
                        // throw new Error(`Failed to ${existingCompany ? 'update' : 'create'} company: ${responseText}`);
                    }

                    const responseData = await response.json();
                    console.log(`Company ${existingCompany ? 'updated' : 'created'} successfully:`, responseData);
                } catch (error) {
                    console.error(`Error ${existingCompany ? 'updating' : 'creating'} company:`, error);
                }
            }

            // Reload the customer list
            loadCustomers();

            // Notify user of successful import
            $('#importModal').modal('hide');
            alert("Customer details imported successfully!");
        };

        reader.readAsBinaryString(file);
    } else {
        alert("Please choose an Excel file.");
    }
});



// Function to determine the role prefix for the customer ID
function getRolePrefix(role) {
    switch (role) {
        case 'Customer':
            return 'CUS';
        case 'Vendor':
            return 'VEN';
        case 'Both':
            return 'CV';
        default:
            return 'CV'; // Default case if role is unrecognized
    }
}

// Load data from localStorage when the page is loaded

/**
 * ----------------------------- script for export-buttons-------------------------------------------------------------------
 */

$(document).ready(function () {

    // Initialize the DataTable
    $('#dataTable').DataTable();

    loadCustomers();

    // Initialize dropdown behavior
    $('#companyVisibility').on('click', function(event) {
        event.preventDefault(); // Prevent default action
        const $dropdownMenu = $(this).next('.dropdown-menu'); // Target the specific dropdown menu
        $('.dropdown-menu').not($dropdownMenu).hide(); // Hide other dropdowns
        $dropdownMenu.toggle(); // Toggle the associated dropdown menu
    });

    // Initialize dropdown behavior
    $('#columnVisibility').on('click', function(event) {
        event.preventDefault(); // Prevent default action
        const $dropdownMenu = $(this).next('.dropdown-menu'); // Target the specific dropdown menu
        $('.dropdown-menu').not($dropdownMenu).hide(); // Hide other dropdowns
        $dropdownMenu.toggle(); // Toggle the associated dropdown menu
    });

    // Close dropdown when clicking outside
    $(document).on('click', function(e) {
        if (!$(e.target).closest('.dropdown').length) {
            $('.dropdown-menu').hide(); // Hide all dropdowns
        }
    });

    // Handle column visibility when the checkbox state changes
    $(document).on('change', '.column-toggle', function () {
        updateColumnVisibility();
    });

    // Initial column visibility update
    updateColumnVisibility();

    // Function to copy visible table data
    $("#copy").click(function () {
        const table = document.getElementById("dataTable");
        const rows = table.querySelectorAll("tr");
        let textToCopy = "";

        // Get headers for copying
        const headers = [];
        table.querySelectorAll("thead th").forEach((header) => {
            if ($(header).is(':visible') && header.innerText.trim() !== 'Actions' && header.innerText.trim() !== 'Image') {
                headers.push(header.innerText.trim());
            }
        });
        textToCopy += headers.join("\t") + "\n"; // Add headers

        // Get data
        rows.forEach(row => {
            const cols = row.querySelectorAll("td, th");

            cols.forEach((col) => {
                const headerText = table.querySelectorAll("th")[col.cellIndex].innerText.trim();

                // Exclude the 'Image' and 'Status' columns and check for visibility
                if (headerText !== 'Image' && headerText !== 'Actions' && $(col).is(':visible')) {
                    textToCopy += col.innerText.trim() + "\t"; // Add a tab between columns
                }
            });
            textToCopy += "\n"; // New line after each row
        });

        navigator.clipboard.writeText(textToCopy.trim());
        alert("Table data copied to clipboard!");
    });

    // Function to export visible table data as CSV
    $("#export-csv").click(function () {
        let csv = [];
        const rows = document.querySelectorAll("#dataTable tr");

        // Get headers for CSV, including "Company Role"
        const headers = [];
        document.querySelectorAll("#dataTable thead th").forEach((header) => {
            const headerText = header.innerText.trim();
            if (headerText !== 'Image' && headerText !== 'Actions' && $(header).is(':visible')) {
                headers.push(headerText);
            }
        });

        // Add "Company Role" as the first column header
        headers.unshift('Company Role');
        csv.push(headers.join(",")); // Add header row

        // Get data
        rows.forEach(row => {
            const cols = row.querySelectorAll("td, th");
            const csvRow = [];

            // Get Company Role based on Company ID
            let companyRole = '';
            const companyIdCol = row.querySelector("td:nth-child(1)"); // Assuming Company ID is in the first column
            if (companyIdCol) {
                const companyId = companyIdCol.innerText.trim();
                if (companyId.includes('CUS')) {
                    companyRole = 'Customer';
                } else if (companyId.includes('VEN')) {
                    companyRole = 'Vendor';
                } else if (companyId.includes('CV')) {
                    companyRole = 'Both';
                }
            }

            // Add the Company Role as the first column
            csvRow.push(companyRole);

            // Add other columns
            cols.forEach((col) => {
                const headerText = document.querySelectorAll("#dataTable th")[col.cellIndex].innerText.trim();
                if (headerText !== 'Image' && headerText !== 'Actions' && $(col).is(':visible')) {
                    csvRow.push(col.innerText.replace(/,/g, " ")); // Replace commas to avoid issues
                }
            });

            if (csvRow.length > 0) {
                csv.push(csvRow.join(",")); // Join the row values with a comma
            }
        });

        const csvString = csv.join("\n");
        const blob = new Blob([csvString], { type: "text/csv;charset=utf-8;" });
        const link = document.createElement("a");
        link.href = URL.createObjectURL(blob);
        link.setAttribute("download", "data.csv");
        link.click();
    });

    // Function to export visible table data as PDF
    $("#export-pdf").click(function () {
        const { jsPDF } = window.jspdf;
        const doc = new jsPDF({ orientation: 'landscape' });
        const table = document.getElementById("dataTable");
        const headers = [];
        const data = [];

        // Get headers from the table, including "Company Role"
        const headerCols = table.querySelectorAll("thead th");
        headerCols.forEach((header) => {
            const headerText = header.innerText.trim();
            if (headerText !== 'Image' && headerText !== 'Actions' && $(header).is(':visible')) {
                headers.push(headerText);
            }
        });

        // Add "Company Role" as the first column header
        headers.unshift('Company Role');

        // Get data from the table rows
        const rows = table.querySelectorAll("tbody tr");
        rows.forEach(row => {
            const rowData = [];
            const cols = row.querySelectorAll("td");

            // Get Company Role based on Company ID
            let companyRole = '';
            const companyIdCol = row.querySelector("td:nth-child(1)"); // Assuming Company ID is in the first column
            if (companyIdCol) {
                const companyId = companyIdCol.innerText.trim();
                if (companyId.includes('CUS')) {
                    companyRole = 'Customer';
                } else if (companyId.includes('VEN')) {
                    companyRole = 'Vendor';
                } else if (companyId.includes('CV')) {
                    companyRole = 'Both';
                }
            }

            // Add the Company Role as the first column
            rowData.push(companyRole);

            // Add other columns
            cols.forEach((col) => {
                const headerText = table.querySelectorAll("thead th")[col.cellIndex].innerText.trim();
                if (headerText !== 'Image' && headerText !== 'Actions' && $(col).is(':visible')) {
                    rowData.push(col.innerText);
                }
            });

            if (rowData.length > 0) {
                data.push(rowData);
            }
        });

        // Generate the PDF using autoTable plugin
        doc.autoTable({
            head: [headers],
            body: data,
            theme: 'grid',
            styles: { fontSize: 10 }
        });

        // Save the PDF
        doc.save('CustomerData.pdf');
    });

    // Function to export visible table data as Excel
    $("#export-excel").click(function () {
        const table = document.getElementById("dataTable");
        const wb = XLSX.utils.book_new();
        const ws = XLSX.utils.aoa_to_sheet([]); // Create empty sheet
    
        // Get headers from the table, including "Company Role" and contact person details
        const headerRow = [];
        const headerCells = table.querySelectorAll("thead th");
    
        headerCells.forEach((headerCell) => {
            const headerText = headerCell.innerText.trim();
            if ((headerText !== 'Company ID' && headerText !== 'Actions' && headerText !== 'Image') && $(headerCell).is(':visible')) {
                headerRow.push(headerText); // Include only relevant headers
            }
        });
    
        // Add "Company Role" as the first column header
        headerRow.unshift('Company Role');
    
        if (headerRow.length > 0) {
            XLSX.utils.sheet_add_aoa(ws, [headerRow], { origin: "A1" });
        }
    
        // Get data from the table rows
        const rows = table.querySelectorAll("tbody tr");
        rows.forEach((row, rowIndex) => {
            const rowData = [];
            const cols = row.querySelectorAll("td");
    
            // Get Company Role based on Company ID
            let companyRole = '';
            const companyIdCol = row.querySelector("td:nth-child(1)"); // Assuming Company ID is in the first column
            if (companyIdCol) {
                const companyId = companyIdCol.innerText.trim();
                if (companyId.includes('CUS')) {
                    companyRole = 'Customer';
                } else if (companyId.includes('VEN')) {
                    companyRole = 'Vendor';
                } else if (companyId.includes('CV')) {
                    companyRole = 'Both';
                }
            }
    
            // Add the Company Role as the first column
            rowData.push(companyRole);
    
            // Add other columns, excluding "Company ID"
            cols.forEach((col) => {
                const headerText = table.querySelectorAll("thead th")[col.cellIndex].innerText.trim();
                if ((headerText !== 'Company ID' && headerText !== 'Actions' && headerText !== 'Image') && $(col).is(':visible')) {
                    rowData.push(col.innerText.trim());
                }
            });
    
            // Add Contact Person details with line breaks for export
            const contactPersonNames = row.querySelector("td.contact-person-names") ? row.querySelector("td.contact-person-names").innerText : '';
            const contactPersonEmails = row.querySelector("td.contact-person-emails") ? row.querySelector("td.contact-person-emails").innerText : '';
            const contactPersonMobiles = row.querySelector("td.contact-person-mobiles") ? row.querySelector("td.contact-person-mobiles").innerText : '';
            const contactPersonDesignations = row.querySelector("td.contact-person-designations") ? row.querySelector("td.contact-person-designations").innerText : '';
    
            // Replace <br> tags with newline (\n) for exporting as line breaks in Excel
            rowData.push(contactPersonNames.replace(/<br>/g, '\n'));
            rowData.push(contactPersonEmails.replace(/<br>/g, '\n'));
            rowData.push(contactPersonMobiles.replace(/<br>/g, '\n'));
            rowData.push(contactPersonDesignations.replace(/<br>/g, '\n'));
    
            // Add row data to sheet
            if (rowData.length > 0) {
                XLSX.utils.sheet_add_aoa(ws, [rowData], { origin: `A${rowIndex + 2}` });
            }
        });
    
        // Add sheet to the workbook and trigger download
        XLSX.utils.book_append_sheet(wb, ws, "Sheet1");
    
        // Trigger Excel file download
        XLSX.writeFile(wb, "customerdata.xlsx");
    });
    

    // Function to print the visible table data
    $("#print").click(function () {
        const table = document.getElementById("dataTable");
        const newWindow = window.open("");
        newWindow.document.write("<html><head><title>Print</title></head><body>");
        newWindow.document.write("<table border='1' style='border-collapse: collapse;'>");

        // Add "Company Role" to header
        newWindow.document.write("<tr>");
        newWindow.document.write("<th>Company Role</th>"); // Add "Company Role" column header

        table.querySelectorAll("thead th").forEach(header => {
            const headerText = header.innerText.trim();
            if (headerText !== 'Image' && headerText !== 'Actions' && $(header).is(':visible')) {
                newWindow.document.write(`<th>${headerText}</th>`);
            }
        });
        newWindow.document.write("</tr>");

        // Get data, including Company Role
        const rows = table.querySelectorAll("tbody tr");
        rows.forEach(row => {
            newWindow.document.write("<tr>");

            // Get Company Role based on Company ID
            const companyIdCol = row.querySelector("td:nth-child(1)");
            let companyRole = '';
            if (companyIdCol) {
                const companyId = companyIdCol.innerText.trim();
                if (companyId.includes('CUS')) {
                    companyRole = 'Customer';
                } else if (companyId.includes('VEN')) {
                    companyRole = 'Vendor';
                } else if (companyId.includes('CV')) {
                    companyRole = 'Both';
                }
            }

            // Add the Company Role
            newWindow.document.write(`<td>${companyRole}</td>`);

            const cols = row.querySelectorAll("td");
            cols.forEach((col) => {
                const headerText = table.querySelectorAll("thead th")[col.cellIndex].innerText.trim();
                if (headerText !== 'Image' && headerText !== 'Actions' && $(col).is(':visible')) {
                    newWindow.document.write(`<td>${col.innerText}</td>`);
                }
            });
            newWindow.document.write("</tr>");
        });

        newWindow.document.write("</table>");
        newWindow.document.write("</body></html>");
        newWindow.document.close();

        // Trigger the print dialog
        newWindow.print();
    });

});

// Function to update column visibility based on checkbox state
function updateColumnVisibility() {
    $(".column-toggle").each(function () {
        const columnIndex = $(this).val();
        const isChecked = $(this).is(":checked");
        const table = $("#dataTable"); // Ensure this ID matches your actual table ID

        // Show or hide the column based on checkbox state
        table.find(`tr`).each(function () {
            $(this).find(`td:eq(${columnIndex}), th:eq(${columnIndex})`).css('display', isChecked ? '' : 'none');
        });
    });
}

async function loadCustomers() {
    try {
        // Fetch customer data from the API
        const response = await fetch('/api/v1/customers');
        if (!response.ok) {
            throw new Error(`Failed to fetch customer data: ${response.statusText}`);
        }
        const customerDataList = await response.json();
        
        let customerTable = $('#dataTable').DataTable();
        customerTable.clear(); // Clear existing data before adding new rows

        // Populate the table with API data
        customerDataList.forEach((customer) => {
            // Combine contact person details for display
            const contactPersonNames = customer.ContactPerson
                ?.map(person => person.name)
                .join('<br>') || '';

            const contactPersonEmails = customer.ContactPerson
                ?.map(person => person.email)
                .join('<br>') || '';

            const contactPersonMobiles = customer.ContactPerson
                ?.map(person => person.mobile)
                .join('<br>') || '';

            const contactPersonDesignations = customer.ContactPerson
                ?.map(person => person.designation)
                .join('<br>') || '';

            // Append the data to the DataTable
            customerTable.row.add([
                customer.companyId1 || '',
                `<canvas class="textCanvas" width="50" height="50" style="display: none;"></canvas>
                <img class="generated-image" src="${customer.companyLogo_url || ''}" alt="Customer Image">`,
                customer.companyName || '',
                customer.acronym || '',
                customer.companyEmail || '',
                customer.companyContact || '',
                customer.secondaryContact || '',
                customer.companyType || '',
                customer.gstNo || '',
                customer.panNo || '',
                customer.msmeNo || '',
                customer.iecNo || '',
                customer.websiteUrl || '',
                customer.billingAddress || '',
                customer.shippingAddress || '',
                customer.bankName || '',
                customer.branchName || '',
                customer.accountNumber || '',
                customer.accountType || '',
                customer.accountHolderName || '',
                customer.accountCurrency || '',
                customer.ifscCode || '',
                customer.swiftCode || '',
                contactPersonNames,
                contactPersonEmails,
                contactPersonMobiles,
                contactPersonDesignations,
                `<div class="row">
                    <button class="emp-action-btn emp-action-edit" data-id="${customer.id}">
                        <span class="icon" title="Edit"><i class="fas fa-pencil-alt"></i></span>
                    </button>
                    <button class="emp-action-btn emp-action-delete" data-id="${customer.id}">
                        <span class="icon" title="Delete"><i class="fas fa-trash-alt"></i></span>
                    </button>
                </div>`
            ]).draw(false); // Use draw(false) to prevent a full redraw
            document.querySelectorAll('.emp-action-edit').forEach((btn) =>
                btn.addEventListener('click', handleEdit)
            );
            document.querySelectorAll('.emp-action-delete').forEach((btn) =>
                btn.addEventListener('click', handleDelete)
            );
        });
    } catch (error) {
        console.error("Error fetching customer data:", error);
        alert(`An error occurred while fetching customer data: ${error.message}`);
    }
}

// Generalized function to generate an ID using API data
async function generateCompanyId(rolePrefix) {
    try {
        const response = await fetch('/api/v1/customers');
        if (!response.ok) {
            throw new Error(`Failed to fetch customer data: ${response.statusText}`);
        }
        const customerDataList = await response.json();

        // Extract the numeric portions of all IDs in the fetched data
        const allIds = customerDataList.map(data => {
            const idParts = data.companyId1?.split('/');
            return idParts && idParts.length === 3 ? parseInt(idParts[2], 10) : 0;
        });

        // Find the largest numeric portion
        const maxIdNumber = Math.max(0, ...allIds);

        // Generate the new ID based on the largest number + 1
        const newIdNumber = (maxIdNumber + 1).toString().padStart(4, '0');

        // Construct and return the new ID with the specified role prefix
        return `UEPL/${rolePrefix}/${newIdNumber}`;
    } catch (error) {
        console.error("Error generating company ID:", error);
        return null;
    }
}

// Function to generate company logo
function generateCompanyLogo(acronym,companyName) {
    const canvas = document.createElement('canvas');
    canvas.width = 50;
    canvas.height = 50;

    const context = canvas.getContext('2d');

    // Array of background colors to cycle through
    const backgroundColors = ['#95b4e6','#95e6cd', '#95e69f', '#b5c979', '#d795e6', '#e69595'];
    let colorNo = 0; // To keep track of the current color

    // Get the next background color from the array
    const backgroundColor = backgroundColors[colorNo];
    colorNo = (colorNo + 1) % backgroundColors.length; // Move to the next color


    // Set the background
    context.fillStyle = backgroundColor; // Background color
    context.fillRect(0, 0, canvas.width, canvas.height);

    // If no acronym is provided, generate initials from the company name
    if (!acronym && companyName) {
        const initials = getInitialsFromName(companyName);

        // Draw the initials
        context.font = '15px Arial'; // Adjust font size
        context.fillStyle = '#FFFFFF'; // Text color (white)
        context.textAlign = 'center';
        context.textBaseline = 'middle';

        // Draw the initials in the center
        context.fillText(initials, canvas.width / 2, canvas.height / 2);
    }else {
        // Draw the acronym
        context.font = '15px Arial'; // Adjust font size
        context.fillStyle = '#FFFFFF'; // Text color (white)
        context.textAlign = 'center';
        context.textBaseline = 'middle';

        // Draw the acronym in the center
        context.fillText(acronym, canvas.width / 2, canvas.height / 2);
    }

    // Return Base64 string
    return canvas.toDataURL('image/png');
}

// Helper function to extract initials from the company name
function getInitialsFromName(name) {
    const words = name.trim().split(' ');
    const initials = words.map(word => word[0].toUpperCase()).join('');
    return initials;
}

// Function to parse multiple contact persons
function parseContactPersons(names, emails, mobiles, designations) {
    const namesArray = names ? names.split(',').map(name => name.trim()) : [];
    const emailsArray = emails ? emails.split(',').map(email => email.trim()) : [];
    const mobilesArray = mobiles ? mobiles.split(',').map(mobile => mobile.trim()) : [];
    const designationsArray = designations ? designations.split(',').map(designation => designation.trim()) : [];

    // Create an array of contact person objects
    return namesArray.map((name, index) => {
        return {
            name: name,
            email: emailsArray[index] || '',
            mobile: mobilesArray[index] || '',
            designation: designationsArray[index] || ''
        };
    });
}

function mergeCompanyData(existingData, newData) {
    newData.forEach(newComp => {
        const existingIndex = existingData.findIndex(comp => comp.companyName === newComp.companyName);
        if (existingIndex !== -1) {
            // Replace existing employee data with the new one
            existingData[existingIndex] = newEmployee;
        } else {
            // Add new employee data
            existingData.push(newEmployee);
        }
    });
    return existingData;
}

/**
 * ----------------------------- script for -------------------------------------------------------------------
 */

document.addEventListener('DOMContentLoaded', function () {
    const companyVisibilityBtn = document.getElementById('companyVisibility');
    const companyDataTable = document.getElementById('dataTable');
    const checkboxes = document.querySelectorAll('.company-toggle');

    // Ensure dropdown toggles correctly
    companyVisibilityBtn.addEventListener('click', function (event) {
        const dropdownMenu = document.querySelector('.dropdown-menu');
        dropdownMenu.classList.toggle('show');
    });

    // Event listener for filtering based on selected roles
    checkboxes.forEach(checkbox => {
        checkbox.addEventListener('change', filterTable);
    });

    function filterTable() {
        const selectedRoles = Array.from(checkboxes)
            .filter(checkbox => checkbox.checked)
            .map(checkbox => checkbox.value);

        const tableRows = companyDataTable.querySelectorAll('tbody tr');

        tableRows.forEach(row => {
            const companyIdCell = row.querySelector('td:first-child'); // Get the first cell (Company ID column)
            if (companyIdCell) {
                const companyId = companyIdCell.textContent.trim();
                const rolePrefix = getRolePrefix(companyId);

                // Show or hide the row based on the selected roles
                if (selectedRoles.includes(rolePrefix)) {
                    row.style.display = ''; // Show the row
                } else {
                    row.style.display = 'none'; // Hide the row
                }
            }
        });
    }

    // Helper function to extract role prefix
    function getRolePrefix(companyId) {
        if (companyId.startsWith('UEPL/CUS/')) {
            return '0'; // Customer role
        } else if (companyId.startsWith('UEPL/VEN/')) {
            return '1'; // Vendor role
        } else if (companyId.startsWith('UEPL/CV/')) {
            return '2'; // Customer & Vendor role
        }
        return ''; // Default for unexpected values
    }
    // Initial filtering to apply the default checked states
    filterTable();
});

function handleEdit(event) {
    const customerId = event.target.closest('button').dataset.id; // Assuming first cell has the customer ID

        // Store the customer ID in session storage
        sessionStorage.setItem('currentCustId', customerId);

        // Redirect to the form page for editing details
        window.location.href = 'customer-details.html';
}

function handleDelete(event) {
    const deleteId = event.target.closest('button').dataset.id;

    let deleteId1;

    const row = event.target.closest('tr');
    deleteId1 = row.cells[0].innerText;

    const confirmDelete = confirm(`Are you sure you want to delete Company ID: ${deleteId1}?`);

    if (confirmDelete) {
        // Proceed with delete operation
        fetch(`/api/v1/customers${deleteId}/`, {
            method: 'DELETE'
        })
        .then(response => {
            if (response.ok) {
                alert('Company deleted successfully.');
                loadCustomers();
            } else {
                alert('Failed to delete company.');
            }
        })
        .catch(error => console.error('Error deleting company:', error));
    }
}