@extends('layouts.dashboard.superadminapp')

@section('title', 'Create Sales Order')

@section('styles')
<style>
    #productTable input.form-control { padding: 0.25rem 0.5rem; font-size: 0.875rem; }
    #productTable textarea.form-control { padding: 0.25rem 0.5rem; font-size: 0.875rem; }
</style>
@endsection

@section('content')
    <!-- Page Heading -->
    <x-layout.page-header title="UEPL | Create Sales Order" icon="shopping-cart" iconColor="success" />

    <!-- Alerts -->
    @if (session('success'))
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fas fa-check-circle mr-2"></i>{{ session('success') }}
            <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                <span aria-hidden="true">&times;</span>
            </button>
        </div>
    @endif
    @if (session('error'))
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-circle mr-2"></i>{{ session('error') }}
            <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                <span aria-hidden="true">&times;</span>
            </button>
        </div>
    @endif

    <!-- Content Card -->
    <div class="card shadow mb-4">
        <div class="card-header py-3 d-flex justify-content-between align-items-center" style="background: linear-gradient(135deg, #4e73df 0%, #224abe 100%);">
            <h6 class="m-0 font-weight-bold text-white">
                <i class="fas fa-file-invoice mr-2"></i> Create Sales Order
            </h6>
            <a href="{{ route('superadmin.sales.salesdetails') }}" class="btn btn-light btn-sm">
                <i class="fa fa-arrow-left mr-1"></i> Back to List
            </a>
        </div>
        <div class="card-body">
            <form id="soForm">
                <div class="row">
                    <div class="col-md-3 mb-3">
                        <label class="small font-weight-bold">Sales Order No <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="sOrderNo" readonly required>
                    </div>
                    <div class="col-md-3 mb-3">
                        <label class="small font-weight-bold">Sales Order Date <span class="text-danger">*</span></label>
                        <input type="date" class="form-control" id="sOrderDate" required>
                    </div>
                    <div class="col-md-3 mb-3">
                        <label class="small font-weight-bold">Quotation</label>
                        <select class="form-control selectpicker" id="quotationSelect" data-live-search="true">
                            <option value="">Select Quotation</option>
                        </select>
                    </div>
                    <div class="col-md-3 mb-3">
                        <label class="small font-weight-bold">Quotation Date</label>
                        <input type="date" class="form-control" id="quotationDate">
                    </div>
                </div>
                <div class="row">
                    <div class="col-md-3 mb-3">
                        <label class="small font-weight-bold">GRN No</label>
                        <input type="text" class="form-control" id="grnNo" placeholder="GRN Number">
                    </div>
                    <div class="col-md-3 mb-3">
                        <label class="small font-weight-bold">Customer Reference No</label>
                        <input type="text" class="form-control" id="custRefNo" placeholder="Customer Ref No">
                    </div>
                    <div class="col-md-6 mb-3">
                        <label class="small font-weight-bold">Customer SO Document</label>
                        <input type="file" class="form-control" id="customerSO" name="customerSO" accept=".pdf">
                    </div>
                </div>

                <hr>

                <h6 class="font-weight-bold text-primary mb-3"><i class="fas fa-user mr-2"></i>Customer Information</h6>
                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label class="small font-weight-bold">Company Name</label>
                        <input type="text" class="form-control" id="soCompanyId" placeholder="Company Name" readonly>
                        <input type="hidden" id="soCompanyName">
                    </div>
                    <div class="col-md-6 mb-3">
                        <label class="small font-weight-bold">Address</label>
                        <input type="text" class="form-control" id="soCompanyAddress" placeholder="Company Address">
                    </div>
                </div>
                <div class="row">
                    <div class="col-md-4 mb-3">
                        <label class="small font-weight-bold">GST No</label>
                        <input type="text" class="form-control" id="soCompanyGSTN" placeholder="GSTIN">
                    </div>
                    <div class="col-md-4 mb-3">
                        <label class="small font-weight-bold">Phone</label>
                        <input type="tel" class="form-control" id="soCompanyPhone" placeholder="Contact Number">
                    </div>
                    <div class="col-md-4 mb-3">
                        <label class="small font-weight-bold">Email</label>
                        <input type="email" class="form-control" id="soCompanyEmail" placeholder="Email Address">
                    </div>
                </div>

                <hr>

                <h6 class="font-weight-bold text-primary mb-3"><i class="fas fa-box mr-2"></i>RFQ & Product Information</h6>
                <div class="row">
                    <div class="col-md-3 mb-3">
                        <label class="small font-weight-bold">RFQ No</label>
                        <input type="text" class="form-control" id="rfqNo" placeholder="RFQ Number">
                    </div>
                    <div class="col-md-3 mb-3">
                        <label class="small font-weight-bold">RFQ Date</label>
                        <input type="date" class="form-control" id="rfqDate">
                    </div>
                    <div class="col-md-3 mb-3">
                        <label class="small font-weight-bold">Part No</label>
                        <input type="text" class="form-control" id="partNo" placeholder="Part Number">
                    </div>
                    <div class="col-md-3 mb-3">
                        <label class="small font-weight-bold">Project/Material No</label>
                        <input type="text" class="form-control" id="projectMaterialNo" placeholder="Project Material No">
                    </div>
                </div>
                <div class="row">
                    <div class="col-md-3 mb-3">
                        <label class="small font-weight-bold">Drawing No</label>
                        <input type="text" class="form-control" id="drawingId" placeholder="Drawing Number">
                    </div>
                    <div class="col-md-3 mb-3">
                        <label class="small font-weight-bold">Drawing Revision</label>
                        <input type="text" class="form-control" id="drawingRevision" placeholder="Revision">
                    </div>
                    <div class="col-md-6 mb-3">
                        <label class="small font-weight-bold">Product Type</label>
                        <input type="text" class="form-control" id="productType" placeholder="Product Type">
                    </div>
                </div>
                <div class="row">
                    <div class="col-md-12 mb-3">
                        <label class="small font-weight-bold">Description / Notes</label>
                        <textarea class="form-control" id="projectDescription" rows="2" placeholder="Project Description"></textarea>
                    </div>
                </div>

                <hr>

                <h6 class="font-weight-bold text-primary mb-3"><i class="fas fa-shipping-fast mr-2"></i>Billing & Shipping</h6>
                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label class="small font-weight-bold">Bill To</label>
                        <textarea class="form-control" id="billAddress" rows="2" placeholder="Billing Address"></textarea>
                    </div>
                    <div class="col-md-6 mb-3">
                        <label class="small font-weight-bold">Ship To</label>
                        <textarea class="form-control" id="shipAddress" rows="2" placeholder="Shipping Address"></textarea>
                    </div>
                </div>

                <hr>

                <h6 class="font-weight-bold text-primary mb-3"><i class="fas fa-list mr-2"></i>Product Items</h6>
                <div class="table-responsive">
                    <table id="productTable" class="table table-bordered table-sm">
                        <thead class="thead-light">
                            <tr>
                                <th width="40">#</th>
                                <th width="90">Material Code</th>
                                <th width="80">HSN Code</th>
                                <th>Description</th>
                                <th>Additional Details</th>
                                <th width="100">Req By Date</th>
                                <th width="60">Qty</th>
                                <th width="60">UOM</th>
                                <th width="80">Unit Rate</th>
                                <th width="60">Disc %</th>
                                <th width="80">Value</th>
                                <th width="60">SGST %</th>
                                <th width="60">CGST %</th>
                                <th width="60">IGST %</th>
                                <th width="90">Amount</th>
                                <th width="40"></th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr>
                                <td class="text-center">1</td>
                                <td><input type="text" class="form-control material-code"></td>
                                <td><input type="text" class="form-control hsn-code"></td>
                                <td><textarea class="form-control description" rows="1"></textarea></td>
                                <td><textarea class="form-control additional-details" rows="1"></textarea></td>
                                <td><input type="date" class="form-control req-by-date"></td>
                                <td><input type="number" class="form-control quantity" min="0"></td>
                                <td><input type="text" class="form-control uom"></td>
                                <td><input type="number" class="form-control rate" step="0.01" min="0"></td>
                                <td><input type="number" class="form-control discount" step="0.01" min="0" max="100"></td>
                                <td><input type="number" class="form-control value" readonly></td>
                                <td><input type="number" class="form-control sgst" step="0.01" min="0"></td>
                                <td><input type="number" class="form-control cgst" step="0.01" min="0"></td>
                                <td><input type="number" class="form-control igst" step="0.01" min="0"></td>
                                <td><input type="number" class="form-control amount" readonly></td>
                                <td class="text-center">
                                    <button type="button" class="btn btn-danger btn-sm" onclick="deleteRow(this)">
                                        <i class="fa fa-trash"></i>
                                    </button>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>

                <div class="mb-3">
                    <button type="button" class="btn btn-primary btn-sm" id="addRowBtn">
                        <i class="fas fa-plus mr-1"></i> Add Row
                    </button>
                </div>

                <hr>

                <h6 class="font-weight-bold text-primary mb-3"><i class="fas fa-calculator mr-2"></i>Summary</h6>
                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label class="small font-weight-bold">Grand Total in Words (INR)</label>
                        <input type="text" class="form-control" id="GT-in-letters" readonly>
                    </div>
                    <div class="col-md-6">
                        <table class="table table-sm table-borderless">
                            <tr>
                                <td class="text-right small font-weight-bold">Sub Total:</td>
                                <td width="150"><input type="text" id="totalAmount" readonly class="form-control form-control-sm text-right"></td>
                            </tr>
                            <tr>
                                <td class="text-right small">SGST Amount:</td>
                                <td><input type="text" id="sgstA" readonly class="form-control form-control-sm text-right"></td>
                            </tr>
                            <tr>
                                <td class="text-right small">CGST Amount:</td>
                                <td><input type="text" id="cgstA" readonly class="form-control form-control-sm text-right"></td>
                            </tr>
                            <tr>
                                <td class="text-right small">IGST Amount:</td>
                                <td><input type="text" id="IgstA" readonly class="form-control form-control-sm text-right"></td>
                            </tr>
                            <tr>
                                <td class="text-right small">Round Up:</td>
                                <td><input type="text" id="roundUp" readonly class="form-control form-control-sm text-right"></td>
                            </tr>
                            <tr class="border-top">
                                <td class="text-right small font-weight-bold text-primary">Grand Total:</td>
                                <td><input type="text" id="grandTotal" readonly class="form-control form-control-sm text-right font-weight-bold"></td>
                            </tr>
                        </table>
                    </div>
                </div>

                <hr>

                <div class="d-flex justify-content-end">
                    <button type="submit" class="btn btn-success btn-sm mr-2" id="submitBtn">
                        <span class="spinner-border spinner-border-sm d-none" role="status"></span>
                        <i class="fas fa-save mr-1"></i> Save
                    </button>
                    <button type="button" class="btn btn-secondary btn-sm mr-2" onclick="resetForm()">
                        <i class="fas fa-times mr-1"></i> Clear
                    </button>
                    <a href="{{ route('superadmin.sales.salesdetails') }}" class="btn btn-outline-secondary btn-sm">
                        <i class="fa fa-arrow-left mr-1"></i> Back
                    </a>
                </div>
            </form>
        </div>
    </div>
@endsection

@section('scripts')
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script>
$(document).ready(function() {
    // Initialize selectpicker
    $('.selectpicker').selectpicker();

    // Set today's date
    document.getElementById('sOrderDate').value = new Date().toISOString().split('T')[0];

    // Generate SO Number
    generateSONumber();

    // Load Quotations
    loadQuotations();

    // Setup initial row event listeners
    setupRowEventListeners(document.querySelector("#productTable tbody tr"));

    // Add Row button
    $('#addRowBtn').click(function() {
        addRow();
    });

    // Form submit
    $('#soForm').on('submit', function(e) {
        e.preventDefault();
        saveSO();
    });

    // Quotation select change
    $('#quotationSelect').on('change', function() {
        loadQuotationDetails(this.value);
    });

    // Calculate totals initially
    calculateAllTotals();
});

// Generate SO Number
async function generateSONumber() {
    try {
        const response = await fetch("{{ url('superadmin/generate-so-id') }}");
        const data = await response.json();
        document.getElementById('sOrderNo').value = data.so_id;
    } catch (error) {
        document.getElementById('sOrderNo').value = 'UEPL/SO/0001';
    }
}

// Load Quotations
async function loadQuotations() {
    try {
        const response = await fetch("{{ url('/superadmin/quotations') }}");
        const quotations = await response.json();
        const select = document.getElementById("quotationSelect");
        quotations.forEach(q => {
            const opt = document.createElement("option");
            opt.value = q.id;
            opt.textContent = q.quotation_no;
            select.appendChild(opt);
        });
        $('.selectpicker').selectpicker('refresh');
    } catch (err) {
        console.error("Failed to load quotations:", err);
    }
}

// Load Quotation Details
async function loadQuotationDetails(quotationId) {
    if (!quotationId) return;
    try {
        const response = await fetch(`{{ url('/superadmin/quotation') }}/${quotationId}`);
        const data = await response.json();
        if (data.error) {
            Swal.fire('Error', data.error, 'error');
            return;
        }

        // Fill form fields
        document.getElementById("quotationDate").value = data.quotation_date || "";
        document.getElementById("soCompanyId").value = data.customer?.company || "";
        document.getElementById("soCompanyName").value = data.customer?.id || "";
        document.getElementById("soCompanyAddress").value = data.address || "";
        document.getElementById("soCompanyGSTN").value = data.gst || "";
        document.getElementById("soCompanyPhone").value = data.phone || "";
        document.getElementById("soCompanyEmail").value = data.email || "";
        document.getElementById("billAddress").value = data.address || "";
        document.getElementById("shipAddress").value = data.address || "";
        document.getElementById("custRefNo").value = data.cust_ref_no || "";
        document.getElementById("rfqNo").value = data.rfq?.rfq_no || "";
        document.getElementById("rfqDate").value = data.rfq_date || "";
        document.getElementById("partNo").value = data.part_no || "";
        document.getElementById("projectMaterialNo").value = data.project_material_no || "";
        document.getElementById("drawingId").value = data.drawing_no || "";
        document.getElementById("drawingRevision").value = data.drawing_rev || "";
        document.getElementById("projectDescription").value = data.note || "";
        document.getElementById("productType").value = data.product_type || "";

        // Fill first row with description and quantity
        const firstRow = document.querySelector("#productTable tbody tr");
        if (firstRow) {
            firstRow.querySelector(".description").value = data.description || "";
            firstRow.querySelector(".quantity").value = data.quantity || "";
            firstRow.querySelector(".uom").value = data.quantity_uom || "";
        }

        calculateAllTotals();
    } catch (err) {
        console.error("Failed to load quotation data:", err);
    }
}

// Add Row
function addRow() {
    const tbody = document.querySelector("#productTable tbody");
    const rowCount = tbody.querySelectorAll('tr').length;
    const newRow = document.createElement("tr");
    newRow.innerHTML = `
        <td class="text-center">${rowCount + 1}</td>
        <td><input type="text" class="form-control material-code"></td>
        <td><input type="text" class="form-control hsn-code"></td>
        <td><textarea class="form-control description" rows="1"></textarea></td>
        <td><textarea class="form-control additional-details" rows="1"></textarea></td>
        <td><input type="date" class="form-control req-by-date"></td>
        <td><input type="number" class="form-control quantity" min="0"></td>
        <td><input type="text" class="form-control uom"></td>
        <td><input type="number" class="form-control rate" step="0.01" min="0"></td>
        <td><input type="number" class="form-control discount" step="0.01" min="0" max="100"></td>
        <td><input type="number" class="form-control value" readonly></td>
        <td><input type="number" class="form-control sgst" step="0.01" min="0"></td>
        <td><input type="number" class="form-control cgst" step="0.01" min="0"></td>
        <td><input type="number" class="form-control igst" step="0.01" min="0"></td>
        <td><input type="number" class="form-control amount" readonly></td>
        <td class="text-center">
            <button type="button" class="btn btn-danger btn-sm" onclick="deleteRow(this)">
                <i class="fa fa-trash"></i>
            </button>
        </td>
    `;
    tbody.appendChild(newRow);
    setupRowEventListeners(newRow);
}

// Delete Row
function deleteRow(button) {
    const row = button.closest("tr");
    row.parentNode.removeChild(row);
    updateRowNumbers();
    calculateAllTotals();
}

// Update Row Numbers
function updateRowNumbers() {
    const rows = document.querySelectorAll("#productTable tbody tr");
    rows.forEach((row, i) => row.cells[0].textContent = i + 1);
}

// Setup Row Event Listeners
function setupRowEventListeners(row) {
    const inputs = ['quantity', 'rate', 'discount', 'sgst', 'cgst', 'igst'];
    inputs.forEach(className => {
        const input = row.querySelector(`.${className}`);
        if (input) {
            input.addEventListener('input', function() {
                calculateRowValue(row);
            });
        }
    });
}

// Calculate Row Value
function calculateRowValue(row) {
    const quantity = parseFloat(row.querySelector('.quantity').value) || 0;
    const rate = parseFloat(row.querySelector('.rate').value) || 0;
    const discount = parseFloat(row.querySelector('.discount').value) || 0;
    const sgst = parseFloat(row.querySelector('.sgst').value) || 0;
    const cgst = parseFloat(row.querySelector('.cgst').value) || 0;
    const igst = parseFloat(row.querySelector('.igst').value) || 0;

    const baseValue = quantity * rate * (1 - discount / 100);
    row.querySelector('.value').value = baseValue.toFixed(2);

    const amount = baseValue * (1 + (sgst + cgst + igst) / 100);
    row.querySelector('.amount').value = amount.toFixed(2);

    calculateAllTotals();
}

// Calculate All Totals
function calculateAllTotals() {
    let subtotal = 0, sgstTotal = 0, cgstTotal = 0, igstTotal = 0;

    document.querySelectorAll("#productTable tbody tr").forEach(row => {
        const val = parseFloat(row.querySelector('.value').value) || 0;
        const sgst = parseFloat(row.querySelector('.sgst').value) || 0;
        const cgst = parseFloat(row.querySelector('.cgst').value) || 0;
        const igst = parseFloat(row.querySelector('.igst').value) || 0;

        subtotal += val;
        sgstTotal += val * sgst / 100;
        cgstTotal += val * cgst / 100;
        igstTotal += val * igst / 100;
    });

    const grandTotalRaw = subtotal + sgstTotal + cgstTotal + igstTotal;
    const roundUp = Math.ceil(grandTotalRaw) - grandTotalRaw;
    const grandTotal = grandTotalRaw + roundUp;

    document.getElementById("totalAmount").value = subtotal.toFixed(2);
    document.getElementById("sgstA").value = sgstTotal.toFixed(2);
    document.getElementById("cgstA").value = cgstTotal.toFixed(2);
    document.getElementById("IgstA").value = igstTotal.toFixed(2);
    document.getElementById("roundUp").value = roundUp.toFixed(2);
    document.getElementById("grandTotal").value = grandTotal.toFixed(2);
    document.getElementById("GT-in-letters").value = numberToWords(Math.round(grandTotal)) + ' Rupees Only';
}

// Get Table Data
function getTableData() {
    const rows = document.querySelectorAll("#productTable tbody tr");
    const data = [];
    rows.forEach(row => {
        data.push({
            slNo: row.cells[0].textContent,
            materialCode: row.querySelector('.material-code').value,
            hsnCode: row.querySelector('.hsn-code').value,
            description: row.querySelector('.description').value,
            additionalDetails: row.querySelector('.additional-details').value,
            reqByDate: row.querySelector('.req-by-date').value,
            quantity: row.querySelector('.quantity').value,
            uom: row.querySelector('.uom').value,
            unitRate: row.querySelector('.rate').value,
            discount: row.querySelector('.discount').value,
            value: row.querySelector('.value').value,
            sgst: row.querySelector('.sgst').value,
            cgst: row.querySelector('.cgst').value,
            igst: row.querySelector('.igst').value,
            amount: row.querySelector('.amount').value
        });
    });
    return data;
}

// Save SO
async function saveSO() {
    const $btn = $('#submitBtn');
    const $spinner = $btn.find('.spinner-border');

    // Validation
    if (!document.getElementById('sOrderNo').value || !document.getElementById('sOrderDate').value) {
        Swal.fire('Error', 'Sales Order No and Date are required!', 'error');
        return;
    }

    $btn.prop('disabled', true);
    $spinner.removeClass('d-none');

    const formData = {
        soNo1: document.getElementById('sOrderNo').value,
        soDate: document.getElementById('sOrderDate').value,
        grnNo1: document.getElementById('grnNo').value,
        custRefSoNo: document.getElementById('custRefNo').value,
        companyName: document.getElementById('soCompanyName').value,
        companyId: document.getElementById('soCompanyId').value,
        soCompanyAddress: document.getElementById('soCompanyAddress').value,
        soCompanyGSTN: document.getElementById('soCompanyGSTN').value,
        soCompanyPhone: document.getElementById('soCompanyPhone').value,
        soCompanyEmail: document.getElementById('soCompanyEmail').value,
        quotationNo2: document.getElementById('quotationSelect').selectedOptions[0]?.text !== 'Select Quotation'
            ? document.getElementById('quotationSelect').selectedOptions[0]?.text
            : null,
        quotationId: document.getElementById('quotationSelect').value,
        quotationDate: document.getElementById('quotationDate').value,
        billAddress: document.getElementById('billAddress').value,
        shipAddress: document.getElementById('shipAddress').value,
        productType: document.getElementById('productType').value,
        totalAmount: document.getElementById('totalAmount').value,
        sgstA: document.getElementById('sgstA').value,
        cgstA: document.getElementById('cgstA').value,
        IgstA: document.getElementById('IgstA').value,
        roundUp: document.getElementById('roundUp').value,
        grandTotal: document.getElementById('grandTotal').value,
        GT_in_letters: document.getElementById('GT-in-letters').value,
        rfqNo: document.getElementById('rfqNo').value,
        rfqDate: document.getElementById('rfqDate').value,
        partNo: document.getElementById('partNo').value,
        projectMaterialNo: document.getElementById('projectMaterialNo').value,
        drawingId: document.getElementById('drawingId').value,
        drawingRevision: document.getElementById('drawingRevision').value,
        projectDescription: document.getElementById('projectDescription').value,
        tableProducts: getTableData()
    };

    try {
        const response = await fetch("{{ route('superadmin.sales.order.store') }}", {
            method: "POST",
            headers: {
                "Content-Type": "application/json",
                "X-CSRF-TOKEN": document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify(formData)
        });

        const data = await response.json();

        if (data.message && data.message.includes('Sales Order saved')) {
            Swal.fire('Success!', data.message, 'success').then(() => {
                window.location.href = "{{ route('superadmin.sales.salesdetails') }}";
            });
        } else {
            Swal.fire('Error!', data.message || data.error || 'Something went wrong.', 'error');
        }
    } catch (err) {
        Swal.fire('Error!', 'Failed to save sales order.', 'error');
        console.error("Save error:", err);
    } finally {
        $btn.prop('disabled', false);
        $spinner.addClass('d-none');
    }
}

// Reset Form
function resetForm() {
    Swal.fire({
        title: 'Clear Form?',
        text: 'Are you sure you want to clear all fields?',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#e74a3b',
        cancelButtonColor: '#858796',
        confirmButtonText: 'Yes, clear it!'
    }).then((result) => {
        if (result.isConfirmed) {
            document.getElementById('soForm').reset();
            generateSONumber();
            document.getElementById('sOrderDate').value = new Date().toISOString().split('T')[0];
            $('.selectpicker').selectpicker('refresh');
            calculateAllTotals();
        }
    });
}

// Number to Words (Indian Format)
function numberToWords(num) {
    if (num === 0) return 'Zero';

    const ones = ['', 'One', 'Two', 'Three', 'Four', 'Five', 'Six', 'Seven', 'Eight', 'Nine',
                  'Ten', 'Eleven', 'Twelve', 'Thirteen', 'Fourteen', 'Fifteen', 'Sixteen',
                  'Seventeen', 'Eighteen', 'Nineteen'];
    const tens = ['', '', 'Twenty', 'Thirty', 'Forty', 'Fifty', 'Sixty', 'Seventy', 'Eighty', 'Ninety'];

    function convertLessThanHundred(n) {
        if (n < 20) return ones[n];
        return tens[Math.floor(n / 10)] + (n % 10 ? ' ' + ones[n % 10] : '');
    }

    function convertLessThanThousand(n) {
        if (n < 100) return convertLessThanHundred(n);
        return ones[Math.floor(n / 100)] + ' Hundred' + (n % 100 ? ' ' + convertLessThanHundred(n % 100) : '');
    }

    if (num < 0) return 'Negative ' + numberToWords(-num);
    if (num < 1000) return convertLessThanThousand(num);
    if (num < 100000) {
        return convertLessThanHundred(Math.floor(num / 1000)) + ' Thousand' +
               (num % 1000 ? ' ' + convertLessThanThousand(num % 1000) : '');
    }
    if (num < 10000000) {
        return convertLessThanHundred(Math.floor(num / 100000)) + ' Lakh' +
               (num % 100000 ? ' ' + numberToWords(num % 100000) : '');
    }
    return convertLessThanHundred(Math.floor(num / 10000000)) + ' Crore' +
           (num % 10000000 ? ' ' + numberToWords(num % 10000000) : '');
}
</script>
@endsection
