{{--
    Select Component

    Usage:
    <x-forms.select label="Customer" name="customer_id" :options="$customers" required />
    <x-forms.select label="Status" name="status" :options="['active' => 'Active', 'inactive' => 'Inactive']" />
    <x-forms.select label="Category" name="category_id" :options="$categories" option-value="id" option-label="name" />

    Props:
    - label: Field label (required)
    - name: Select name attribute (required)
    - options: Array or Collection of options (required)
    - value: Selected value (optional)
    - placeholder: Placeholder option text (default: '-- Select --')
    - required: Is field required (default: false)
    - disabled: Is field disabled (default: false)
    - multiple: Allow multiple selection (default: false)
    - searchable: Enable Select2 search (default: false)
    - optionValue: Key for option value in objects/collections (default: 'id')
    - optionLabel: Key for option label in objects/collections (default: 'name')
    - col: Bootstrap column class (default: col-md-6)
    - help: Help text below select (optional)
--}}

@props([
    'label',
    'name',
    'options',
    'value' => null,
    'placeholder' => '-- Select --',
    'required' => false,
    'disabled' => false,
    'multiple' => false,
    'searchable' => false,
    'optionValue' => 'id',
    'optionLabel' => 'name',
    'col' => 'col-md-6',
    'help' => null
])

@php
    $inputId = $name . '_' . uniqid();
    $selectedValue = old($name, $value);
    $hasError = $errors->has($name);

    // Convert Collection to array if needed
    if ($options instanceof \Illuminate\Support\Collection) {
        $options = $options->toArray();
    }
@endphp

<div class="{{ $col }} mb-3">
    <label for="{{ $inputId }}" class="form-label">
        {{ $label }}
        @if($required)
        <span class="text-danger">*</span>
        @endif
    </label>

    <select
        id="{{ $inputId }}"
        name="{{ $name }}{{ $multiple ? '[]' : '' }}"
        class="form-control {{ $searchable ? 'select2' : '' }} {{ $hasError ? 'is-invalid' : '' }}"
        {{ $required ? 'required' : '' }}
        {{ $disabled ? 'disabled' : '' }}
        {{ $multiple ? 'multiple' : '' }}
        {{ $attributes }}
    >
        @if(!$multiple && $placeholder)
        <option value="">{{ $placeholder }}</option>
        @endif

        @foreach($options as $key => $option)
            @php
                // Handle different option formats
                if (is_object($option)) {
                    $optValue = $option->{$optionValue} ?? $option->id ?? $key;
                    $optLabel = $option->{$optionLabel} ?? $option->name ?? $option;
                } elseif (is_array($option)) {
                    $optValue = $option[$optionValue] ?? $option['id'] ?? $key;
                    $optLabel = $option[$optionLabel] ?? $option['name'] ?? $option;
                } else {
                    // Simple key => value array
                    $optValue = $key;
                    $optLabel = $option;
                }

                // Check if selected
                if ($multiple && is_array($selectedValue)) {
                    $isSelected = in_array($optValue, $selectedValue);
                } else {
                    $isSelected = $selectedValue == $optValue;
                }
            @endphp
            <option value="{{ $optValue }}" {{ $isSelected ? 'selected' : '' }}>
                {{ $optLabel }}
            </option>
        @endforeach
    </select>

    @if($hasError)
    <div class="invalid-feedback d-block">
        {{ $errors->first($name) }}
    </div>
    @endif

    @if($help)
    <small class="form-text text-muted">{{ $help }}</small>
    @endif
</div>

@if($searchable)
@push('scripts')
<script>
$(document).ready(function() {
    $('#{{ $inputId }}').select2({
        theme: 'bootstrap4',
        placeholder: '{{ $placeholder }}',
        allowClear: true
    });
});
</script>
@endpush
@endif
