{{--
    File Upload Component

    Usage:
    <x-forms.file-upload label="Document" name="document" />
    <x-forms.file-upload label="Images" name="images" multiple accept="image/*" />
    <x-forms.file-upload label="Excel File" name="excel" accept=".xlsx,.xls,.csv" />

    Props:
    - label: Field label (required)
    - name: Input name attribute (required)
    - accept: Accepted file types (optional)
    - multiple: Allow multiple files (default: false)
    - required: Is field required (default: false)
    - disabled: Is field disabled (default: false)
    - maxSize: Max file size in MB (default: 10)
    - preview: Show file preview (default: false)
    - col: Bootstrap column class (default: col-md-6)
    - help: Help text below input (optional)
--}}

@props([
    'label',
    'name',
    'accept' => null,
    'multiple' => false,
    'required' => false,
    'disabled' => false,
    'maxSize' => 10,
    'preview' => false,
    'col' => 'col-md-6',
    'help' => null
])

@php
    $inputId = $name . '_' . uniqid();
    $hasError = $errors->has($name);
@endphp

<div class="{{ $col }} mb-3">
    <label for="{{ $inputId }}" class="form-label">
        {{ $label }}
        @if($required)
        <span class="text-danger">*</span>
        @endif
    </label>

    <div class="custom-file">
        <input
            type="file"
            id="{{ $inputId }}"
            name="{{ $name }}{{ $multiple ? '[]' : '' }}"
            class="custom-file-input {{ $hasError ? 'is-invalid' : '' }}"
            {{ $required ? 'required' : '' }}
            {{ $disabled ? 'disabled' : '' }}
            {{ $multiple ? 'multiple' : '' }}
            @if($accept) accept="{{ $accept }}" @endif
            {{ $attributes }}
        >
        <label class="custom-file-label" for="{{ $inputId }}">
            Choose file{{ $multiple ? '(s)' : '' }}...
        </label>
    </div>

    @if($hasError)
    <div class="invalid-feedback d-block">
        {{ $errors->first($name) }}
    </div>
    @endif

    @if($help)
    <small class="form-text text-muted">{{ $help }}</small>
    @else
    <small class="form-text text-muted">
        Max file size: {{ $maxSize }}MB
        @if($accept)
        | Accepted: {{ $accept }}
        @endif
    </small>
    @endif

    @if($preview)
    <div id="{{ $inputId }}_preview" class="mt-2"></div>
    @endif
</div>

@push('scripts')
<script>
$(document).ready(function() {
    // Update label with selected filename(s)
    $('#{{ $inputId }}').on('change', function() {
        var files = this.files;
        var label = $(this).next('.custom-file-label');

        if (files.length === 0) {
            label.text('Choose file{{ $multiple ? "(s)" : "" }}...');
        } else if (files.length === 1) {
            label.text(files[0].name);
        } else {
            label.text(files.length + ' files selected');
        }

        @if($preview)
        // Show preview for images
        var previewDiv = $('#{{ $inputId }}_preview');
        previewDiv.empty();

        Array.from(files).forEach(function(file) {
            if (file.type.match('image.*')) {
                var reader = new FileReader();
                reader.onload = function(e) {
                    previewDiv.append(
                        '<img src="' + e.target.result + '" class="img-thumbnail mr-2 mb-2" style="max-height: 100px;" alt="File preview">'
                    );
                };
                reader.readAsDataURL(file);
            }
        });
        @endif
    });
});
</script>
@endpush
