<?php

namespace App\Services\Purchase;

use App\Services\BaseService;
use App\Repositories\PurchaseQuotationRepository;
use App\Models\PurchaseQuotation;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class PurchaseQuotationService extends BaseService
{
    protected PurchaseQuotationRepository $quotationRepository;

    public function __construct(PurchaseQuotationRepository $repository)
    {
        parent::__construct($repository);
        $this->quotationRepository = $repository;
    }

    /**
     * Get quotations with vendor
     */
    public function getAllWithVendor(): Collection
    {
        return $this->quotationRepository->getWithVendor();
    }

    /**
     * Get quotations by vendor
     */
    public function getByVendor(int $vendorId): Collection
    {
        return $this->quotationRepository->getByVendor($vendorId);
    }

    /**
     * Get quotations available for PO
     */
    public function getAvailableForPurchaseOrder(): Collection
    {
        return $this->quotationRepository->getAvailableForPurchaseOrder();
    }

    /**
     * Create quotation from RFQ
     */
    public function createFromRfq(int $rfqId, array $data): PurchaseQuotation
    {
        return DB::transaction(function () use ($rfqId, $data) {
            $data['purchase_rfq_id'] = $rfqId;
            $data['quotation_no'] = $this->generateQuotationNumber();

            $quotation = $this->create($data);
            $this->logAction('Purchase Quotation created from RFQ', [
                'id' => $quotation->id,
                'rfq_id' => $rfqId,
            ]);

            return $quotation;
        });
    }

    /**
     * Create quotation with items
     */
    public function createWithItems(array $data, array $items): PurchaseQuotation
    {
        return DB::transaction(function () use ($data, $items) {
            $data['quotation_no'] = $this->generateQuotationNumber();
            $quotation = $this->create($data);

            foreach ($items as $item) {
                $item['purchase_quotation_id'] = $quotation->id;
                $quotation->items()->create($item);
            }

            $this->logAction('Purchase Quotation created', ['id' => $quotation->id]);
            return $quotation->fresh(['items']);
        });
    }

    /**
     * Generate quotation number
     */
    public function generateQuotationNumber(): string
    {
        $year = date('Y');
        return nextNumber("PQT/{$year}", 'purchase_quotations', 'quotation_no', 2);
    }

    /**
     * Approve quotation
     */
    public function approve(int $id): PurchaseQuotation
    {
        return DB::transaction(function () use ($id) {
            $quotation = $this->findByIdOrFail($id);
            $quotation->update(['status' => 'approved']);
            $this->logAction('Purchase Quotation approved', ['id' => $id]);
            return $quotation->fresh();
        });
    }

    /**
     * Reject quotation
     */
    public function reject(int $id, ?string $remarks = null): PurchaseQuotation
    {
        return DB::transaction(function () use ($id, $remarks) {
            $quotation = $this->findByIdOrFail($id);
            $quotation->update([
                'status' => 'rejected',
                'remarks' => $remarks,
            ]);
            $this->logAction('Purchase Quotation rejected', ['id' => $id]);
            return $quotation->fresh();
        });
    }
}
