<?php

namespace App\Services\Employee;

use App\Services\BaseService;
use App\Repositories\EmployeeRepository;
use App\Models\EmployeeDetails;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class EmployeeService extends BaseService
{
    protected EmployeeRepository $employeeRepository;

    public function __construct(EmployeeRepository $repository)
    {
        parent::__construct($repository);
        $this->employeeRepository = $repository;
    }

    /**
     * Get active employees
     */
    public function getActive(): Collection
    {
        return $this->employeeRepository->getActive();
    }

    /**
     * Get employees by department
     */
    public function getByDepartment(string $department): Collection
    {
        return $this->employeeRepository->getByDepartment($department);
    }

    /**
     * Get operators
     */
    public function getOperators(): Collection
    {
        return $this->employeeRepository->getOperators();
    }

    /**
     * Search employees
     */
    public function search(string $term): Collection
    {
        return $this->employeeRepository->search($term);
    }

    /**
     * Generate employee ID
     */
    public function generateEmployeeId(): string
    {
        return nextNumber('EMP', 'employee_details', 'employee_id', 1);
    }

    /**
     * Create employee with profile picture
     */
    public function createWithProfilePicture(array $data, $profilePicture = null): EmployeeDetails
    {
        return DB::transaction(function () use ($data, $profilePicture) {
            $data['employee_id'] = $data['employee_id'] ?? $this->generateEmployeeId();

            if ($profilePicture) {
                $data['profile_picture'] = $profilePicture->store('employee-photos', 'public');
            }

            $employee = $this->create($data);
            $this->logAction('Employee created', ['id' => $employee->id]);
            return $employee;
        });
    }

    /**
     * Update with profile picture
     */
    public function updateWithProfilePicture(int $id, array $data, $profilePicture = null): EmployeeDetails
    {
        return DB::transaction(function () use ($id, $data, $profilePicture) {
            $employee = $this->findByIdOrFail($id);

            if ($profilePicture) {
                // Delete old picture
                if ($employee->profile_picture) {
                    Storage::disk('public')->delete($employee->profile_picture);
                }
                $data['profile_picture'] = $profilePicture->store('employee-photos', 'public');
            }

            $employee->update($data);
            $this->logAction('Employee updated', ['id' => $id]);
            return $employee->fresh();
        });
    }

    /**
     * Delete profile picture
     */
    public function deleteProfilePicture(int $id): bool
    {
        $employee = $this->findByIdOrFail($id);
        if ($employee->profile_picture) {
            Storage::disk('public')->delete($employee->profile_picture);
            $employee->update(['profile_picture' => null]);
            return true;
        }
        return false;
    }

    /**
     * Get departments list
     */
    public function getDepartments(): array
    {
        return EmployeeDetails::distinct()
            ->pluck('department')
            ->filter()
            ->values()
            ->toArray();
    }

    /**
     * Get designations list
     */
    public function getDesignations(): array
    {
        return EmployeeDetails::distinct()
            ->pluck('designation')
            ->filter()
            ->values()
            ->toArray();
    }
}
