<?php

namespace App\Services\Employee;

use App\Services\BaseService;
use App\Repositories\AttendanceRepository;
use App\Models\EmployeeAttendance;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class AttendanceService extends BaseService
{
    protected AttendanceRepository $attendanceRepository;

    public function __construct(AttendanceRepository $repository)
    {
        parent::__construct($repository);
        $this->attendanceRepository = $repository;
    }

    /**
     * Get attendance with employee
     */
    public function getAllWithEmployee(): Collection
    {
        return $this->attendanceRepository->getWithEmployee();
    }

    /**
     * Get by employee
     */
    public function getByEmployee(int $employeeId): Collection
    {
        return $this->attendanceRepository->getByEmployee($employeeId);
    }

    /**
     * Get by date
     */
    public function getByDate(string $date): Collection
    {
        return $this->attendanceRepository->getByDate($date);
    }

    /**
     * Get by date range
     */
    public function getByDateRange(int $employeeId, string $start, string $end): Collection
    {
        return $this->attendanceRepository->getByDateRange($employeeId, $start, $end);
    }

    /**
     * Get monthly attendance
     */
    public function getMonthly(int $employeeId, int $month, int $year): Collection
    {
        return $this->attendanceRepository->getMonthly($employeeId, $month, $year);
    }

    /**
     * Mark attendance
     */
    public function markAttendance(array $data): EmployeeAttendance
    {
        return DB::transaction(function () use ($data) {
            // Check if already marked
            $existing = EmployeeAttendance::where('employee_id', $data['employee_id'])
                ->where('date', $data['date'])
                ->first();

            if ($existing) {
                $existing->update($data);
                return $existing->fresh();
            }

            return $this->create($data);
        });
    }

    /**
     * Bulk mark attendance
     */
    public function bulkMark(string $date, array $attendanceData): int
    {
        return DB::transaction(function () use ($date, $attendanceData) {
            $count = 0;
            foreach ($attendanceData as $data) {
                $data['date'] = $date;
                $this->markAttendance($data);
                $count++;
            }
            return $count;
        });
    }

    /**
     * Get attendance summary for month
     */
    public function getMonthlySummary(int $employeeId, int $month, int $year): array
    {
        $attendance = $this->getMonthly($employeeId, $month, $year);

        return [
            'total_days' => $attendance->count(),
            'present' => $attendance->where('status', 'present')->count(),
            'absent' => $attendance->where('status', 'absent')->count(),
            'half_day' => $attendance->where('status', 'half_day')->count(),
            'leave' => $attendance->where('status', 'leave')->count(),
        ];
    }
}
