<?php

namespace App\Services\Cache;

/**
 * Cache Keys Constants
 *
 * Centralized cache key constants for consistent
 * cache key management across the application.
 */
final class CacheKeys
{
    // ──────────────────────────────────────────────────────────────────────────
    // MASTER DATA KEYS
    // ──────────────────────────────────────────────────────────────────────────

    public const CUSTOMERS_ALL = 'master:customers:all';
    public const CUSTOMERS_ACTIVE = 'master:customers:active';
    public const VENDORS_ALL = 'master:vendors:all';
    public const VENDORS_ACTIVE = 'master:vendors:active';
    public const COMPANY_TYPES = 'master:company_types';
    public const ACCOUNT_TYPES = 'master:account_types';
    public const PRODUCT_TYPES = 'master:product_types';
    public const PROCESSES = 'master:processes';
    public const MACHINES = 'master:machines';
    public const MACHINES_ACTIVE = 'master:machines:active';
    public const RFQ_TYPES = 'master:rfq_types';

    // ──────────────────────────────────────────────────────────────────────────
    // DASHBOARD KEYS
    // ──────────────────────────────────────────────────────────────────────────

    public const DASHBOARD_ADMIN_STATS = 'dashboard:admin:stats';
    public const DASHBOARD_ADMIN_CHARTS = 'dashboard:admin:charts';
    public const DASHBOARD_MANAGER_STATS = 'dashboard:manager:stats';
    public const DASHBOARD_MANAGER_CHARTS = 'dashboard:manager:charts';
    public const DASHBOARD_SUPERADMIN_STATS = 'dashboard:superadmin:stats';
    public const DASHBOARD_SUPERADMIN_CHARTS = 'dashboard:superadmin:charts';

    // ──────────────────────────────────────────────────────────────────────────
    // COUNTS KEYS
    // ──────────────────────────────────────────────────────────────────────────

    public const COUNT_PENDING_RFQS = 'counts:pending_rfqs';
    public const COUNT_PENDING_QUOTATIONS = 'counts:pending_quotations';
    public const COUNT_PENDING_SALES_ORDERS = 'counts:pending_sales_orders';
    public const COUNT_UNPAID_INVOICES = 'counts:unpaid_invoices';
    public const COUNT_PENDING_POS = 'counts:pending_pos';
    public const COUNT_PENDING_GRNS = 'counts:pending_grns';
    public const COUNT_ACTIVE_ROUTE_CARDS = 'counts:active_route_cards';
    public const COUNT_PENDING_QC = 'counts:pending_qc';
    public const COUNT_LOW_STOCK_ITEMS = 'counts:low_stock_items';

    // ──────────────────────────────────────────────────────────────────────────
    // REPORT KEYS
    // ──────────────────────────────────────────────────────────────────────────

    public const REPORT_SALES_SUMMARY = 'report:sales:summary';
    public const REPORT_PURCHASE_SUMMARY = 'report:purchase:summary';
    public const REPORT_PRODUCTION_STATUS = 'report:production:status';
    public const REPORT_INVENTORY_STATUS = 'report:inventory:status';
    public const REPORT_AGING_RECEIVABLES = 'report:aging:receivables';
    public const REPORT_AGING_PAYABLES = 'report:aging:payables';

    // ──────────────────────────────────────────────────────────────────────────
    // KEY GENERATORS
    // ──────────────────────────────────────────────────────────────────────────

    /**
     * Generate key for single customer/vendor
     */
    public static function customer(int $id): string
    {
        return "customer_vendor:{$id}";
    }

    /**
     * Generate key for single RFQ
     */
    public static function rfq(int $id): string
    {
        return "rfq:{$id}";
    }

    /**
     * Generate key for single quotation
     */
    public static function quotation(int $id): string
    {
        return "quotation:{$id}";
    }

    /**
     * Generate key for single sales order
     */
    public static function salesOrder(int $id): string
    {
        return "sales_order:{$id}";
    }

    /**
     * Generate key for single invoice
     */
    public static function invoice(int $id): string
    {
        return "invoice:{$id}";
    }

    /**
     * Generate key for single purchase order
     */
    public static function purchaseOrder(int $id): string
    {
        return "purchase_order:{$id}";
    }

    /**
     * Generate key for single GRN
     */
    public static function grn(int $id): string
    {
        return "grn:{$id}";
    }

    /**
     * Generate key for single route card
     */
    public static function routeCard(int $id): string
    {
        return "route_card:{$id}";
    }

    /**
     * Generate key for single employee
     */
    public static function employee(int $id): string
    {
        return "employee:{$id}";
    }

    /**
     * Generate key for report with date range
     */
    public static function reportWithDates(string $type, string $from, string $to): string
    {
        return "report:{$type}:{$from}:{$to}";
    }

    /**
     * Generate key for paginated list
     */
    public static function paginatedList(string $entity, int $page, int $perPage, array $filters = []): string
    {
        $filterHash = md5(json_encode($filters));
        return "{$entity}:list:page_{$page}:per_{$perPage}:{$filterHash}";
    }
}
