<?php

namespace App\Repositories;

use App\Models\EmployeeSalaryStructure;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Collection;

class SalaryStructureRepository extends BaseRepository
{
    protected function resolveModel(): Model
    {
        return new EmployeeSalaryStructure();
    }

    /**
     * Get all with employee details
     */
    public function getWithEmployee(): Collection
    {
        return $this->with(['employee'])
            ->orderBy('created_at', 'desc')
            ->all();
    }

    /**
     * Get active structures
     */
    public function getActive(): Collection
    {
        return $this->where('status', EmployeeSalaryStructure::STATUS_ACTIVE)
            ->with(['employee'])
            ->all();
    }

    /**
     * Get by employee ID
     */
    public function getByEmployee(int $employeeId): ?EmployeeSalaryStructure
    {
        return $this->where('employee_id', $employeeId)
            ->orderBy('id', 'desc')
            ->all()
            ->first();
    }

    /**
     * Get active structure for employee
     */
    public function getActiveForEmployee(int $employeeId): ?EmployeeSalaryStructure
    {
        $query = $this->getQuery();
        $result = $query->where('employee_id', $employeeId)
            ->where('status', EmployeeSalaryStructure::STATUS_ACTIVE)
            ->first();
        $this->resetQuery();
        return $result;
    }

    /**
     * Get currently effective structure for employee
     */
    public function getCurrentlyEffective(int $employeeId): ?EmployeeSalaryStructure
    {
        $today = now()->toDateString();
        $query = $this->getQuery();
        $result = $query->where('employee_id', $employeeId)
            ->where('status', EmployeeSalaryStructure::STATUS_ACTIVE)
            ->where(function ($q) use ($today) {
                $q->whereNull('effective_from')
                  ->orWhere('effective_from', '<=', $today);
            })
            ->where(function ($q) use ($today) {
                $q->whereNull('effective_to')
                  ->orWhere('effective_to', '>=', $today);
            })
            ->first();
        $this->resetQuery();
        return $result;
    }

    /**
     * Get employees without salary structure
     */
    public function getEmployeesWithoutStructure(): Collection
    {
        $query = $this->getModel()->newQuery();
        $existingEmployeeIds = $query->pluck('employee_id')->toArray();

        return \App\Models\EmployeeDetails::whereNotIn('id', $existingEmployeeIds)
            ->where('status', 'active')
            ->get();
    }

    /**
     * Check if employee has salary structure
     */
    public function hasStructure(int $employeeId): bool
    {
        $query = $this->getQuery();
        $result = $query->where('employee_id', $employeeId)->exists();
        $this->resetQuery();
        return $result;
    }

    /**
     * Deactivate structure by ID
     */
    public function deactivate(int $id): bool
    {
        $structure = $this->findOrFail($id);
        return $structure->update(['status' => EmployeeSalaryStructure::STATUS_INACTIVE]);
    }

    /**
     * Deactivate all structures for an employee
     */
    public function deactivateForEmployee(int $employeeId): int
    {
        return $this->model->where('employee_id', $employeeId)
            ->where('status', EmployeeSalaryStructure::STATUS_ACTIVE)
            ->update(['status' => EmployeeSalaryStructure::STATUS_INACTIVE]);
    }
}
