<?php

namespace App\Repositories;

use App\Models\RouteCard;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Collection;

class RouteCardRepository extends BaseRepository
{
    protected function resolveModel(): Model
    {
        return new RouteCard();
    }

    /**
     * Get route cards with all relationships
     */
    public function getWithDetails(): Collection
    {
        return $this->with(['processes', 'salesOrder', 'customer'])
            ->orderBy('created_at', 'desc')
            ->all();
    }

    /**
     * Get route cards by status
     */
    public function getByStatus(string $status): Collection
    {
        return $this->where('status', $status)
            ->with(['processes'])
            ->orderBy('created_at', 'desc')
            ->all();
    }

    /**
     * Get route cards by sales order
     */
    public function getBySalesOrder(int $salesOrderId): Collection
    {
        return $this->where('sales_order_id', $salesOrderId)->all();
    }

    /**
     * Get approved route cards ready for planning
     */
    public function getApproved(): Collection
    {
        return $this->where('status', 'approved')
            ->with(['processes', 'salesOrder'])
            ->orderBy('priority', 'desc')
            ->all();
    }

    /**
     * Get route cards in production
     */
    public function getInProduction(): Collection
    {
        return $this->where('status', 'in_production')
            ->with(['processes'])
            ->orderBy('created_at', 'desc')
            ->all();
    }

    /**
     * Get planned route cards
     */
    public function getPlanned(): Collection
    {
        return $this->where('status', 'planned')
            ->with(['processes'])
            ->orderBy('priority', 'desc')
            ->all();
    }

    /**
     * Get route cards ready for production
     */
    public function getReadyForProduction(): Collection
    {
        return $this->where('status', 'ready_for_production')
            ->with(['processes', 'salesOrder'])
            ->orderBy('priority', 'desc')
            ->all();
    }
}
