<?php

namespace App\Repositories;

use App\Models\QualityControl;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Collection;

class QualityControlRepository extends BaseRepository
{
    protected function resolveModel(): Model
    {
        return new QualityControl();
    }

    /**
     * Get QC records with route card
     */
    public function getWithDetails(): Collection
    {
        return $this->with(['routeCard', 'routeCardProcess'])
            ->orderBy('created_at', 'desc')
            ->all();
    }

    /**
     * Get QC by status
     */
    public function getByStatus(string $status): Collection
    {
        return $this->where('status', $status)
            ->orderBy('created_at', 'desc')
            ->all();
    }

    /**
     * Get pending QC inspections
     */
    public function getPending(): Collection
    {
        return $this->where('status', QualityControl::STATUS_PENDING)
            ->with(['routeCard'])
            ->orderBy('created_at', 'asc')
            ->all();
    }

    /**
     * Get QC by route card
     */
    public function getByRouteCard(int $routeCardId): Collection
    {
        return $this->where('route_card_id', $routeCardId)->all();
    }

    /**
     * Get failed/rejected QC inspections
     */
    public function getFailed(): Collection
    {
        return $this->where('status', QualityControl::STATUS_REJECTED)
            ->with(['routeCard'])
            ->orderBy('created_at', 'desc')
            ->all();
    }

    /**
     * Get QC by process
     */
    public function getByProcess(int $processId): Collection
    {
        return $this->where('process_id', $processId)
            ->orderBy('sl_no', 'asc')
            ->all();
    }

    /**
     * Get approved QC inspections
     */
    public function getApproved(): Collection
    {
        return $this->where('status', QualityControl::STATUS_APPROVED)
            ->with(['routeCard', 'inspector'])
            ->orderBy('inspected_at', 'desc')
            ->all();
    }
}
