<?php

namespace App\Repositories;

use App\Models\EmployeePayroll;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Collection;

class PayrollRepository extends BaseRepository
{
    protected function resolveModel(): Model
    {
        return new EmployeePayroll();
    }

    /**
     * Get payroll with employee details
     */
    public function getWithEmployee(): Collection
    {
        return $this->with(['employee'])
            ->orderBy('month_year', 'desc')
            ->all();
    }

    /**
     * Get payroll by employee
     */
    public function getByEmployee(int $employeeId): Collection
    {
        return $this->where('employee_id', $employeeId)
            ->orderBy('month_year', 'desc')
            ->all();
    }

    /**
     * Get payroll by month
     */
    public function getByMonth(string $monthYear): Collection
    {
        return $this->where('month_year', $monthYear)
            ->with(['employee'])
            ->all();
    }

    /**
     * Get pending payroll for approval
     */
    public function getPending(): Collection
    {
        return $this->where('status', 'pending')
            ->with(['employee'])
            ->orderBy('created_at', 'desc')
            ->all();
    }

    /**
     * Get approved payroll
     */
    public function getApproved(string $monthYear): Collection
    {
        return $this->where('month_year', $monthYear)
            ->where('status', 'approved')
            ->with(['employee'])
            ->all();
    }

    /**
     * Get payroll by department
     */
    public function getByDepartment(string $department, string $monthYear): Collection
    {
        return $this->where('month_year', $monthYear)
            ->with(['employee' => function ($q) use ($department) {
                $q->where('department', $department);
            }])
            ->all();
    }
}
