<?php

namespace App\Repositories;

use App\Models\EmployeeDailyReport;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Collection;

class DailyReportRepository extends BaseRepository
{
    protected function resolveModel(): Model
    {
        return new EmployeeDailyReport();
    }

    /**
     * Get reports with employee details
     */
    public function getWithEmployee(): Collection
    {
        return $this->with(['employeeDetails'])
            ->orderBy('date', 'desc')
            ->all();
    }

    /**
     * Get reports by employee
     */
    public function getByEmployee(string $employeeId): Collection
    {
        return $this->where('employee_id', $employeeId)
            ->orderBy('date', 'desc')
            ->all();
    }

    /**
     * Get reports by date
     */
    public function getByDate(string $date): Collection
    {
        return $this->where('date', $date)
            ->with(['employeeDetails'])
            ->all();
    }

    /**
     * Get reports for date range
     */
    public function getByDateRange(string $start, string $end): Collection
    {
        return $this->whereBetween('date', [$start, $end])
            ->with(['employeeDetails'])
            ->orderBy('date', 'asc')
            ->all();
    }

    /**
     * Get employee reports for date range
     */
    public function getEmployeeReportsInRange(string $employeeId, string $start, string $end): Collection
    {
        $query = $this->getQuery();
        $query->where('employee_id', $employeeId)
            ->whereBetween('date', [$start, $end])
            ->orderBy('date', 'asc');
        $result = $query->get();
        $this->resetQuery();
        return $result;
    }

    /**
     * Get monthly reports
     */
    public function getMonthly(string $employeeId, int $month, int $year): Collection
    {
        $query = $this->getQuery();
        $query->where('employee_id', $employeeId)
            ->whereMonth('date', $month)
            ->whereYear('date', $year)
            ->orderBy('date', 'asc');
        $result = $query->get();
        $this->resetQuery();
        return $result;
    }

    /**
     * Get reports with overtime
     */
    public function getWithOvertime(int $month, int $year): Collection
    {
        $query = $this->getQuery();
        $query->where('over_time', '>', 0)
            ->whereMonth('date', $month)
            ->whereYear('date', $year)
            ->with(['employeeDetails'])
            ->orderBy('over_time', 'desc');
        $result = $query->get();
        $this->resetQuery();
        return $result;
    }

    /**
     * Get manual entries
     */
    public function getManualEntries(string $start, string $end): Collection
    {
        $query = $this->getQuery();
        $query->where('man_entry', true)
            ->whereBetween('date', [$start, $end])
            ->with(['employeeDetails'])
            ->orderBy('date', 'desc');
        $result = $query->get();
        $this->resetQuery();
        return $result;
    }

    /**
     * Find by employee and date
     */
    public function findByEmployeeAndDate(string $employeeId, string $date): ?EmployeeDailyReport
    {
        $query = $this->getQuery();
        $result = $query->where('employee_id', $employeeId)
            ->where('date', $date)
            ->first();
        $this->resetQuery();
        return $result;
    }
}
