<?php

namespace App\Repositories\Contracts;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Pagination\LengthAwarePaginator;

interface RepositoryInterface
{
    /**
     * Get all records
     */
    public function all(): Collection;

    /**
     * Get all records with specific columns
     */
    public function allWith(array $columns = ['*']): Collection;

    /**
     * Find a record by ID
     */
    public function find(int $id): ?Model;

    /**
     * Find a record by ID or fail
     */
    public function findOrFail(int $id): Model;

    /**
     * Find records by a specific field
     */
    public function findBy(string $field, mixed $value): Collection;

    /**
     * Find first record by a specific field
     */
    public function findFirstBy(string $field, mixed $value): ?Model;

    /**
     * Create a new record
     */
    public function create(array $data): Model;

    /**
     * Update a record by ID
     */
    public function update(int $id, array $data): Model;

    /**
     * Delete a record by ID
     */
    public function delete(int $id): bool;

    /**
     * Paginate records
     */
    public function paginate(int $perPage = 15, array $columns = ['*']): LengthAwarePaginator;

    /**
     * Get query builder with eager loaded relations
     */
    public function with(array $relations): static;

    /**
     * Get records ordered by a field
     */
    public function orderBy(string $column, string $direction = 'asc'): static;

    /**
     * Get count of records
     */
    public function count(): int;

    /**
     * Check if record exists
     */
    public function exists(int $id): bool;
}
