<?php

namespace App\Repositories;

use App\Models\EmployeeAttendance;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Collection;

class AttendanceRepository extends BaseRepository
{
    protected function resolveModel(): Model
    {
        return new EmployeeAttendance();
    }

    /**
     * Get attendance with employee details
     */
    public function getWithEmployee(): Collection
    {
        return $this->with(['employee'])
            ->orderBy('date', 'desc')
            ->all();
    }

    /**
     * Get attendance by employee
     */
    public function getByEmployee(int $employeeId): Collection
    {
        return $this->where('employee_id', $employeeId)
            ->orderBy('date', 'desc')
            ->all();
    }

    /**
     * Get attendance by date
     */
    public function getByDate(string $date): Collection
    {
        return $this->where('date', $date)
            ->with(['employee'])
            ->all();
    }

    /**
     * Get attendance for date range
     */
    public function getByDateRange(int $employeeId, string $start, string $end): Collection
    {
        return $this->where('employee_id', $employeeId)
            ->whereBetween('date', [$start, $end])
            ->orderBy('date', 'asc')
            ->all();
    }

    /**
     * Get monthly attendance summary
     */
    public function getMonthly(int $employeeId, int $month, int $year): Collection
    {
        $query = $this->getQuery();
        $query->where('employee_id', $employeeId)
            ->whereMonth('date', $month)
            ->whereYear('date', $year)
            ->orderBy('date', 'asc');
        $result = $query->get();
        $this->resetQuery();
        return $result;
    }
}
