<?php

namespace App\Models;

use App\Models\Traits\HasAuditFields;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class SalesPayment extends Model
{
    use SoftDeletes, HasAuditFields;

    /**
     * Payment status constants
     */
    public const STATUS_PENDING = 'pending';
    public const STATUS_RECONCILED = 'reconciled';
    public const STATUS_CLEARED = 'cleared';
    public const STATUS_DISPUTED = 'disputed';

    /**
     * Valid payment methods
     */
    public const PAYMENT_METHODS = [
        'cash' => 'Cash',
        'cheque' => 'Cheque',
        'bank_transfer' => 'Bank Transfer',
        'upi' => 'UPI',
        'card' => 'Card',
        'rtgs' => 'RTGS',
        'neft' => 'NEFT',
        'imps' => 'IMPS',
    ];

    protected $table = 'sales_payments';

    protected $fillable = [
        'payment_no',
        'invoice_id',
        'invoice_no',
        'party_id',
        'payment_date',
        'payment_amount',
        'payment_method',
        'reference_no',
        'notes',
        'status',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'payment_date' => 'date',
        'payment_amount' => 'decimal:2',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // ──────────────────────────────────────────────────────────────────────────
    // RELATIONSHIPS
    // ──────────────────────────────────────────────────────────────────────────

    public function invoice()
    {
        return $this->belongsTo(TaxInvoice::class, 'invoice_id');
    }

    public function party()
    {
        return $this->belongsTo(CustomerVendor::class, 'party_id');
    }

    // ──────────────────────────────────────────────────────────────────────────
    // SCOPES
    // ──────────────────────────────────────────────────────────────────────────

    public function scopePending($query)
    {
        return $query->where('status', self::STATUS_PENDING);
    }

    public function scopeReconciled($query)
    {
        return $query->where('status', self::STATUS_RECONCILED);
    }

    public function scopeCleared($query)
    {
        return $query->where('status', self::STATUS_CLEARED);
    }

    public function scopeDisputed($query)
    {
        return $query->where('status', self::STATUS_DISPUTED);
    }

    // ──────────────────────────────────────────────────────────────────────────
    // STATUS HELPERS
    // ──────────────────────────────────────────────────────────────────────────

    public function isPending(): bool
    {
        return $this->status === self::STATUS_PENDING;
    }

    public function isReconciled(): bool
    {
        return $this->status === self::STATUS_RECONCILED;
    }

    public function isCleared(): bool
    {
        return $this->status === self::STATUS_CLEARED;
    }

    public function isDisputed(): bool
    {
        return $this->status === self::STATUS_DISPUTED;
    }

    /**
     * Get valid payment methods for dropdowns
     */
    public static function getPaymentMethods(): array
    {
        return self::PAYMENT_METHODS;
    }
}
