<?php

namespace App\Models;

use App\Models\Traits\HasAuditFields;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Process extends Model
{
    use HasFactory, SoftDeletes, HasAuditFields;

    protected $table = 'processes';

    /**
     * Status constants
     */
    public const STATUS_ACTIVE = 'active';
    public const STATUS_INACTIVE = 'inactive';

    protected $fillable = [
        'product_type_id',
        'name',
        'order',
        'status',
        'created_by',
        'updated_by',
    ];

    protected $attributes = [
        'status' => self::STATUS_ACTIVE,
    ];

    protected $casts = [
        'order' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // ──────────────────────────────────────────────────────────────────────────
    // RELATIONSHIPS
    // ──────────────────────────────────────────────────────────────────────────

    /**
     * Get the product type for this process
     */
    public function productType()
    {
        return $this->belongsTo(ProductType::class);
    }

    /**
     * Get machines assigned to this process
     */
    public function machines()
    {
        return $this->belongsToMany(Machines::class, 'process_machine', 'process_id', 'machine_id')
            ->withTimestamps();
    }

    /**
     * Get operators for machines in this process
     */
    public function operators()
    {
        // Fetch operators who are assigned to machines used in this process
        return $this->belongsToMany(EmployeeDetails::class, 'machine_operator', 'machine_id', 'employee_id')
            ->join('process_machine', 'machine_operator.machine_id', '=', 'process_machine.machine_id')
            ->where('process_machine.process_id', $this->id)
            ->withTimestamps();
    }

    // ──────────────────────────────────────────────────────────────────────────
    // SCOPES
    // ──────────────────────────────────────────────────────────────────────────

    /**
     * Filter by active status
     */
    public function scopeActive(Builder $query): Builder
    {
        return $query->where('status', self::STATUS_ACTIVE);
    }

    /**
     * Filter by product type
     */
    public function scopeOfProductType(Builder $query, int $productTypeId): Builder
    {
        return $query->where('product_type_id', $productTypeId);
    }

    /**
     * Order by process order
     */
    public function scopeOrdered(Builder $query): Builder
    {
        return $query->orderBy('order', 'asc');
    }

    /**
     * Search by name
     */
    public function scopeSearch(Builder $query, string $term): Builder
    {
        return $query->where('name', 'like', "%{$term}%");
    }

    // ──────────────────────────────────────────────────────────────────────────
    // HELPERS
    // ──────────────────────────────────────────────────────────────────────────

    /**
     * Get all valid statuses
     */
    public static function getStatuses(): array
    {
        return [
            self::STATUS_ACTIVE,
            self::STATUS_INACTIVE,
        ];
    }

    /**
     * Check if active
     */
    public function isActive(): bool
    {
        return $this->status === self::STATUS_ACTIVE;
    }
}
