<?php

namespace App\Models;

use App\Models\Traits\HasAuditFields;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class MachineLog extends Model
{
    use HasFactory, SoftDeletes, HasAuditFields;

    protected $table = 'machine_logs';

    /**
     * Activity type constants
     */
    public const ACTIVITY_STARTED = 'started';
    public const ACTIVITY_STOPPED = 'stopped';
    public const ACTIVITY_BREAKDOWN = 'breakdown';
    public const ACTIVITY_MAINTENANCE = 'maintenance';
    public const ACTIVITY_REPAIR = 'repair';

    protected $fillable = [
        'machine_id',
        'activity',
        'reason',
        'remarks',
        'start_time',
        'end_time',
        'logged_by',
        'logged_at',
        'reported_by',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'start_time' => 'datetime',
        'end_time' => 'datetime',
        'logged_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // ──────────────────────────────────────────────────────────────────────────
    // RELATIONSHIPS
    // ──────────────────────────────────────────────────────────────────────────

    /**
     * Get the machine this log belongs to
     */
    public function machine()
    {
        return $this->belongsTo(Machines::class, 'machine_id');
    }

    /**
     * Get the user who reported this log
     */
    public function reporter()
    {
        return $this->belongsTo(EmployeeDetails::class, 'reported_by');
    }

    /**
     * Get the user who logged this activity
     */
    public function logger()
    {
        return $this->belongsTo(User::class, 'logged_by');
    }

    // ──────────────────────────────────────────────────────────────────────────
    // SCOPES
    // ──────────────────────────────────────────────────────────────────────────

    /**
     * Filter by machine
     */
    public function scopeForMachine(Builder $query, int $machineId): Builder
    {
        return $query->where('machine_id', $machineId);
    }

    /**
     * Filter by activity type
     */
    public function scopeOfActivity(Builder $query, string $activity): Builder
    {
        return $query->where('activity', $activity);
    }

    /**
     * Order by logged time (newest first)
     */
    public function scopeLatest(Builder $query): Builder
    {
        return $query->orderBy('logged_at', 'desc');
    }

    // ──────────────────────────────────────────────────────────────────────────
    // HELPERS
    // ──────────────────────────────────────────────────────────────────────────

    /**
     * Get all valid activity types
     */
    public static function getActivityTypes(): array
    {
        return [
            self::ACTIVITY_STARTED,
            self::ACTIVITY_STOPPED,
            self::ACTIVITY_BREAKDOWN,
            self::ACTIVITY_MAINTENANCE,
            self::ACTIVITY_REPAIR,
        ];
    }
}
