<?php

namespace App\Http\Resources;

use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\ResourceCollection;

/**
 * Base Collection
 *
 * Provides standardized collection response format
 * with pagination meta and additional data.
 */
abstract class BaseCollection extends ResourceCollection
{
    /**
     * Additional meta data
     */
    protected array $additionalMeta = [];

    /**
     * Add additional meta data
     */
    public function withAdditionalMeta(array $meta): self
    {
        $this->additionalMeta = array_merge($this->additionalMeta, $meta);
        return $this;
    }

    /**
     * Get additional data
     */
    public function with(Request $request): array
    {
        return [
            'success' => true,
            'meta' => array_merge([
                'total' => $this->collection->count(),
            ], $this->additionalMeta),
        ];
    }

    /**
     * Customize pagination information
     */
    public function paginationInformation($request, $paginated, $default): array
    {
        return [
            'pagination' => [
                'total' => $paginated['total'] ?? 0,
                'per_page' => $paginated['per_page'] ?? 15,
                'current_page' => $paginated['current_page'] ?? 1,
                'last_page' => $paginated['last_page'] ?? 1,
                'from' => $paginated['from'] ?? null,
                'to' => $paginated['to'] ?? null,
                'has_more' => ($paginated['current_page'] ?? 1) < ($paginated['last_page'] ?? 1),
            ],
        ];
    }
}
