<?php

namespace App\Http\Requests\Sales;

use App\Http\Requests\BaseFormRequest;
use Illuminate\Validation\Validator;

class StoreInvoiceRequest extends BaseFormRequest
{
    public function rules(): array
    {
        return [
            'invoice_date' => 'required|date',
            'customer_name' => 'required|exists:customer_vendors,id',
            'sales_order_no' => 'nullable|string|max:100',
            'sales_order_date' => 'nullable|date',
            'po_no' => 'nullable|string|max:100',
            'po_date' => 'nullable|date',
            'challan_no' => 'nullable|string|max:100',
            'challan_date' => 'nullable|date',
            'part_no' => 'nullable|string|max:100',
            'project_material_no' => 'nullable|string|max:100',
            'drawing_no' => 'nullable|string|max:100',
            'drawing_rev' => 'nullable|string|max:50',
            'quantity' => 'required|numeric|min:0.01',
            'quantity_uom' => 'required|string|max:20',
            'description' => 'nullable|string|max:1000',
            'product_type' => 'nullable|string|max:100',
            'note' => 'nullable|string|max:2000',
            'sub_total' => 'required|numeric|min:0',
            'discount' => 'nullable|numeric|min:0|max:100',
            // GST validation - rates must be valid percentages
            'cgst' => 'nullable|numeric|min:0|max:14', // Max CGST is 14%
            'sgst' => 'nullable|numeric|min:0|max:14', // Max SGST is 14%
            'igst' => 'nullable|numeric|min:0|max:28', // Max IGST is 28%
            'cgst_amount' => 'nullable|numeric|min:0',
            'sgst_amount' => 'nullable|numeric|min:0',
            'igst_amount' => 'nullable|numeric|min:0',
            'round_up' => 'nullable|numeric|min:-1|max:1', // Round up should be small
            'grand_total' => 'required|numeric|min:0',
            'due_date' => 'nullable|date|after_or_equal:invoice_date',
            'items' => 'nullable|array',
            'items.*.description' => 'required_with:items|string',
            'items.*.hsn_code' => 'nullable|string|max:20',
            'items.*.quantity' => 'required_with:items|numeric|min:0',
            'items.*.rate' => 'required_with:items|numeric|min:0',
            'items.*.amount' => 'required_with:items|numeric|min:0',
        ];
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator(Validator $validator): void
    {
        $validator->after(function (Validator $validator) {
            $this->validateGstRules($validator);
            $this->validateTotalCalculation($validator);
        });
    }

    /**
     * Validate GST business rules
     */
    protected function validateGstRules(Validator $validator): void
    {
        $cgst = floatval($this->input('cgst', 0));
        $sgst = floatval($this->input('sgst', 0));
        $igst = floatval($this->input('igst', 0));

        // IGST and CGST+SGST are mutually exclusive
        if ($igst > 0 && ($cgst > 0 || $sgst > 0)) {
            $validator->errors()->add('igst', 'IGST cannot be applied with CGST/SGST. Use either IGST (inter-state) or CGST+SGST (intra-state).');
        }

        // CGST and SGST must be equal
        if ($cgst > 0 || $sgst > 0) {
            if (abs($cgst - $sgst) > 0.01) {
                $validator->errors()->add('cgst', 'CGST and SGST rates must be equal.');
            }
        }

        // Validate GST rates are standard rates (0, 5, 12, 18, 28)
        $validGstRates = [0, 2.5, 5, 6, 9, 12, 14, 18, 28];

        if ($igst > 0 && !in_array($igst, $validGstRates)) {
            $validator->errors()->add('igst', 'IGST rate must be a valid GST rate (0%, 5%, 12%, 18%, or 28%).');
        }
    }

    /**
     * Validate grand total calculation
     */
    protected function validateTotalCalculation(Validator $validator): void
    {
        $subTotal = floatval($this->input('sub_total', 0));
        $discount = floatval($this->input('discount', 0));
        $cgstAmount = floatval($this->input('cgst_amount', 0));
        $sgstAmount = floatval($this->input('sgst_amount', 0));
        $igstAmount = floatval($this->input('igst_amount', 0));
        $roundUp = floatval($this->input('round_up', 0));
        $grandTotal = floatval($this->input('grand_total', 0));

        $calculatedTotal = $subTotal - $discount + $cgstAmount + $sgstAmount + $igstAmount + $roundUp;

        // Allow small floating point differences (up to 1 rupee)
        if (abs($calculatedTotal - $grandTotal) > 1) {
            $validator->errors()->add('grand_total', 'Grand total calculation mismatch. Expected: ₹' . number_format($calculatedTotal, 2));
        }
    }

    public function attributes(): array
    {
        return [
            'invoice_date' => 'Invoice Date',
            'customer_name' => 'Customer',
            'challan_no' => 'Challan Number',
            'due_date' => 'Due Date',
            'cgst' => 'CGST Rate',
            'sgst' => 'SGST Rate',
            'igst' => 'IGST Rate',
            'cgst_amount' => 'CGST Amount',
            'sgst_amount' => 'SGST Amount',
            'igst_amount' => 'IGST Amount',
        ];
    }

    public function messages(): array
    {
        return [
            'cgst.max' => 'CGST rate cannot exceed 14%.',
            'sgst.max' => 'SGST rate cannot exceed 14%.',
            'igst.max' => 'IGST rate cannot exceed 28%.',
            'discount.max' => 'Discount cannot exceed 100%.',
        ];
    }
}
