@push('scripts')

<!-- Select2 JS -->
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>

<script>
    // Function to handle AJAX page navigation
    function loadPurchasePage(url) {
        // Extract the query string to ensure filters are preserved
        const urlObj = new URL(url, window.location.origin);
        
        $.ajax({
            url: url,
            type: 'GET',
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            },
            beforeSend: function() {
                $('#pills-invoice').html('<div class="text-center p-5"><i class="fas fa-spinner fa-spin fa-3x"></i><p class="mt-3">Loading...</p></div>');
            },
            success: function(response) {
                if (response.success) {
                    $('#pills-invoice').html(response.html);
                    
                    // Reinitialize DataTable
                    if ($.fn.DataTable.isDataTable('#purchaseInvoiceTable')) {
                        $('#purchaseInvoiceTable').DataTable().destroy();
                    }
                    $('#purchaseInvoiceTable').DataTable({
                        dom: 'Bfrtip',
                        buttons: [
                            'copy', 'csv', 'excel', 'pdf', 'print', 'colvis'
                        ],
                        responsive: true,
                        paging: false,
                        info: false
                    });
                    
                    // Don't update URL for pagination - just load content
                }
            },
            error: function(xhr) {
                console.error('Error loading page:', xhr);
                $('#pills-invoice').html('<div class="alert alert-danger">Error loading data. Please try again.</div>');
            }
        });
    }

    $(document).ready(function() {

        // Add purchase invoice button - Open modal
        $('#addPurchaseInvoiceBtn').on('click', function() {
            $('#purchaseInvoiceForm')[0].reset();
            $('#purchaseInvoiceForm').removeData('edit-id');
            $('#purchaseInvoiceForm').removeData('edit-row');
            $('#purchaseInvoiceForm').attr('action', '{{ route('superadmin.purchase.v1.store') }}');
            $('#modalTitle').text('Add New Purchase Invoice');
            isFormDirty = false;
            $('#purchaseInvoiceModal').modal('show');
        });


        // View purchase invoice details
        $(document).on('click', '.view-purchase-invoice', function() {
            let id = $(this).data('id');
            $('#viewPurchaseInvoiceBody').html(
                '<div class="text-center"><i class="fas fa-spinner fa-spin"></i> Loading...</div>'
            );
            $('#viewPurchaseInvoiceModal').modal('show');
        });

        // Delete purchase invoice
        $(document).on('click', '.delete-purchase-invoice', function() {
            let id = $(this).data('id');
            Swal.fire({
                title: 'Are you sure?',
                text: "You want to delete this purchase invoice!",
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#d33',
                cancelButtonColor: '#3085d6',
                confirmButtonText: 'Yes, Delete!',
                reverseButtons: true
            }).then((result) => {
                if (result.isConfirmed) {
                    $.ajax({
                        url: '{{ url("/superadmin/purchase/destroy") }}/' + id,
                        type: 'DELETE',
                        data: {
                            _token: '{{ csrf_token() }}',
                            _method: 'DELETE'
                        },
                        success: function(response) {
                            Swal.fire('Deleted!', 'Purchase Invoice deleted successfully.', 'success');
                            location.reload();
                        },
                        error: function(xhr) {
                            Swal.fire('Error', 'Error occurred while deleting.', 'error');
                            console.error(xhr.responseText);
                        }
                    });
                }
            });
        });

        // Create credit note from purchase invoice
        $(document).on('click', '.create-credit-note', function() {
            let invoiceNo = $(this).data('invoice');
            let vendorId = $(this).data('vendor');
            let vendorName = $(this).data('vendor-name');
            
            $('#creditNoteForm')[0].reset();
            $('#creditNoteForm').removeData('edit-id');
            $('#creditNoteModalLabel').text('Create Credit Note for ' + invoiceNo);
            
            // Get next credit note number
            $.get('{{ route('superadmin.purchase.v1.v1.creditNotes.nextNumber') }}', function(response) {
                if (response.success) {
                    $('#credit_note_no').val(response.credit_note_no);
                }
            });
            
            // Pre-fill data
            $('#credit_note_date').val(new Date().toISOString().split('T')[0]);
            $('#credit_note_invoice').val(invoiceNo);
            $('#credit_note_party').val(vendorId);
            
            // Switch to credit notes tab and show modal
            $('#pills-credit-tab').click();
            setTimeout(function() {
                $('#creditNoteModal').modal('show');
            }, 100);
        });

        // Credit Notes functionality
        let creditNoteTable;

        // Initialize credit notes tab when clicked
        $(document).on('click', '#pills-credit-tab', function() {
            if (!creditNoteTable) {
                initializeCreditNotesTable();
            }
        });

        function initializeCreditNotesTable() {
            creditNoteTable = $('#creditTable').DataTable({
                ajax: {
                    url: '{{ route('superadmin.purchase.v1.v1.creditNotes') }}',
                    type: 'GET',
                    data: function(d) {
                        d.party_id = $('#credit-party').val();
                        d.date_range = $('.credit-date-range').val();
                        d.credit_note_no = $('.credit-note-filter').val();
                    }
                },
                columns: [
                    { data: 'credit_note_no' },
                    { 
                        data: 'date',
                        render: function(data) {
                            return new Date(data).toLocaleDateString();
                        }
                    },
                    { data: 'party' },
                    { data: 'invoice_no' },
                    { 
                        data: 'amount',
                        render: function(data) {
                            return '₹' + parseFloat(data).toFixed(2);
                        }
                    },
                    { data: 'reason' },
                    {
                        data: null,
                        render: function(data) {
                            return `
                                <button class="btn btn-sm btn-outline-primary edit-credit-note" data-id="${data.id}">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-danger delete-credit-note" data-id="${data.id}">
                                    <i class="fas fa-trash"></i>
                                </button>
                            `;
                        }
                    }
                ],
                dom: 'Bfrtip',
                buttons: [
                    'copy', 'csv', 'excel', 'pdf', 'print', 'colvis'
                ],
                responsive: true,
                order: [[1, 'desc']]
            });
        }

        // Add credit note button
        $(document).on('click', '#addCreditNoteBtn', function() {
            $('#creditNoteForm')[0].reset();
            $('#creditNoteForm').removeData('edit-id');
            $('#creditNoteModalLabel').text('Add Credit Note');
            
            // Get next credit note number
            $.get('{{ route('superadmin.purchase.v1.v1.creditNotes.nextNumber') }}', function(response) {
                if (response.success) {
                    $('#credit_note_no').val(response.credit_note_no);
                }
            });
            
            // Set today's date
            $('#credit_note_date').val(new Date().toISOString().split('T')[0]);
            
            $('#creditNoteModal').modal('show');
        });

        // Edit credit note
        $(document).on('click', '.edit-credit-note', function() {
            let id = $(this).data('id');
            
            $.get('{{ url("/superadmin/purchase/v1/credit-notes/get") }}/' + id, function(response) {
                if (response.success) {
                    let data = response.data;
                    $('#creditNoteForm').data('edit-id', id);
                    $('#creditNoteModalLabel').text('Edit Credit Note');
                    
                    $('#credit_note_no').val(data.credit_note_no);
                    $('#credit_note_date').val(data.date);
                    $('#credit_note_party').val(data.party_id);
                    $('#credit_note_invoice').val(data.invoice_no);
                    $('#credit_note_amount').val(data.amount);
                    $('#credit_note_reason').val(data.reason);
                    $('#credit_note_description').val(data.description);
                    
                    $('#creditNoteModal').modal('show');
                }
            });
        });

        // Delete credit note
        $(document).on('click', '.delete-credit-note', function() {
            let id = $(this).data('id');
            
            Swal.fire({
                title: 'Are you sure?',
                text: "You want to delete this credit note!",
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#d33',
                cancelButtonColor: '#3085d6',
                confirmButtonText: 'Yes, Delete!',
                reverseButtons: true
            }).then((result) => {
                if (result.isConfirmed) {
                    $.ajax({
                        url: '{{ url("/superadmin/purchase/v1/credit-notes/destroy") }}/' + id,
                        type: 'DELETE',
                        data: {
                            _token: '{{ csrf_token() }}'
                        },
                        success: function(response) {
                            if (response.success) {
                                Swal.fire('Deleted!', response.message, 'success');
                                creditNoteTable.ajax.reload();
                            } else {
                                Swal.fire('Error', response.message, 'error');
                            }
                        },
                        error: function(xhr) {
                            Swal.fire('Error', 'Error occurred while deleting.', 'error');
                        }
                    });
                }
            });
        });

        // Submit credit note form
        $(document).on('submit', '#creditNoteForm', function(e) {
            e.preventDefault();
            
            let formData = new FormData(this);
            let editId = $(this).data('edit-id');
            let url, method;
            
            if (editId) {
                url = '{{ url("/superadmin/purchase/v1/credit-notes/update") }}/' + editId;
                method = 'PUT';
                formData.append('_method', 'PUT');
            } else {
                url = '{{ route('superadmin.purchase.v1.v1.creditNotes.store') }}';
                method = 'POST';
            }
            
            $.ajax({
                url: url,
                type: method,
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        $('#creditNoteModal').modal('hide');
                        Swal.fire('Success!', response.message, 'success');
                        if (creditNoteTable) {
                            creditNoteTable.ajax.reload();
                        }
                    } else {
                        Swal.fire('Error', response.message, 'error');
                    }
                },
                error: function(xhr) {
                    let errorMessage = 'An error occurred';
                    if (xhr.responseJSON && xhr.responseJSON.errors) {
                        errorMessage = Object.values(xhr.responseJSON.errors).flat().join('<br>');
                    } else if (xhr.responseJSON && xhr.responseJSON.message) {
                        errorMessage = xhr.responseJSON.message;
                    }
                    Swal.fire('Error', errorMessage, 'error');
                }
            });
        });

        // Credit notes filter functionality
        $(document).on('click', '#applyCreditFilters', function() {
            if (creditNoteTable) {
                creditNoteTable.ajax.reload();
            }
        });

        $(document).on('click', '#resetCreditFilters', function() {
            $('#credit-party').val('');
            $('.credit-date-range').val('');
            $('.credit-note-filter').val('');
            if (creditNoteTable) {
                creditNoteTable.ajax.reload();
            }
        });

        // Initialize date range picker for credit notes
        $('.credit-date-range').daterangepicker({
            autoUpdateInput: false,
            locale: {
                cancelLabel: 'Clear'
            }
        });

        $('.credit-date-range').on('apply.daterangepicker', function(ev, picker) {
            $(this).val(picker.startDate.format('YYYY-MM-DD') + ' - ' + picker.endDate.format('YYYY-MM-DD'));
        });

        $('.credit-date-range').on('cancel.daterangepicker', function(ev, picker) {
            $(this).val('');
        });

        // Aging Report functionality
        let agingChart;
        let agingTable;
        let currentAgingData = null;

        // Initialize aging report when tab is clicked
        $(document).on('click', '#pills-aging-tab', function() {
            if (!agingTable) {
                initializeAgingReport();
            }
        });

        function initializeAgingReport() {
            loadAgingReport();
        }

        function loadAgingReport() {
            const asOfDate = $('#aging-as-of-date').val();
            const partyId = $('#aging-party').val();
            const reportType = $('#aging-report-type').val();

            // Show loading indicator
            showAgingLoading();

            $.ajax({
                url: '{{ route('superadmin.purchase.v1.agingReport') }}',
                type: 'GET',
                data: {
                    as_of_date: asOfDate,
                    party_id: partyId,
                    report_type: reportType
                },
                success: function(response) {
                    if (response.success) {
                        currentAgingData = response;
                        updateAgingSummaryCards(response.totals);
                        updateAgingChart(response.data, response.totals);
                        updateAgingTable(response.data, response.totals);
                    } else {
                        Swal.fire('Error', 'Failed to load aging report', 'error');
                    }
                },
                error: function(xhr) {
                    console.error('Error loading aging report:', xhr);
                    Swal.fire('Error', 'Failed to load aging report', 'error');
                },
                complete: function() {
                    hideAgingLoading();
                }
            });
        }

        function showAgingLoading() {
            $('#agingTable tbody').html('<tr><td colspan="8" class="text-center"><i class="fas fa-spinner fa-spin"></i> Loading...</td></tr>');
        }

        function hideAgingLoading() {
            // Loading hidden when table is populated
        }

        function updateAgingSummaryCards(totals) {
            $('#totalOutstanding').text('₹' + formatNumber(totals.total_due));
            $('#currentDue').text('₹' + formatNumber(totals.current));
            $('#overdueAmount').text('₹' + formatNumber(totals['1_30'] + totals['31_60'] + totals['61_90'] + totals.over_90));
            $('#criticalOverdue').text('₹' + formatNumber(totals.over_90));
        }

        function updateAgingChart(data, totals) {
            const ctx = document.getElementById('agingChart').getContext('2d');
            
            // Destroy existing chart
            if (agingChart) {
                agingChart.destroy();
            }

            agingChart = new Chart(ctx, {
                type: 'bar',
                data: {
                    labels: ['Current', '1-30 Days', '31-60 Days', '61-90 Days', '>90 Days'],
                    datasets: [{
                        label: 'Outstanding Amount',
                        data: [
                            totals.current,
                            totals['1_30'],
                            totals['31_60'],
                            totals['61_90'],
                            totals.over_90
                        ],
                        backgroundColor: [
                            '#28a745',
                            '#ffc107',
                            '#fd7e14',
                            '#dc3545',
                            '#6f42c1'
                        ],
                        borderColor: [
                            '#1e7e34',
                            '#e0a800',
                            '#e55100',
                            '#bd2130',
                            '#59359a'
                        ],
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        title: {
                            display: true,
                            text: 'Aging Analysis - Outstanding Amounts by Period',
                            font: { size: 16, weight: 'bold' }
                        },
                        legend: {
                            display: false
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    return 'Amount: ₹' + formatNumber(context.parsed.y);
                                }
                            }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                callback: function(value) {
                                    return '₹' + formatNumber(value);
                                }
                            }
                        }
                    }
                }
            });
        }

        function updateAgingTable(data, totals) {
            // Destroy existing DataTable
            if (agingTable) {
                agingTable.destroy();
            }

            const tbody = $('#agingTable tbody');
            tbody.empty();

            // Populate table rows
            data.forEach(function(row) {
                const tr = $(`
                    <tr>
                        <td>${row.vendor_name || 'Unknown Vendor'}</td>
                        <td class="text-end bucket-current">₹${formatNumber(row.current)}</td>
                        <td class="text-end bucket-1-30">₹${formatNumber(row['1_30'])}</td>
                        <td class="text-end bucket-31-60">₹${formatNumber(row['31_60'])}</td>
                        <td class="text-end bucket-61-90">₹${formatNumber(row['61_90'])}</td>
                        <td class="text-end bucket-over-90">₹${formatNumber(row.over_90)}</td>
                        <td class="text-end fw-bold">₹${formatNumber(row.total_due)}</td>
                        <td class="text-center">
                            <button class="btn btn-sm btn-outline-info view-detailed-aging" 
                                    data-vendor-id="${row.vendor_id}" 
                                    data-vendor-name="${row.vendor_name}"
                                    title="View Details">
                                <i class="fas fa-eye"></i>
                            </button>
                        </td>
                    </tr>
                `);
                tbody.append(tr);
            });

            // Add totals footer
            const footer = $('#agingTableFooter');
            footer.html(`
                <tr>
                    <th class="fw-bold">TOTAL</th>
                    <th class="text-end fw-bold">₹${formatNumber(totals.current)}</th>
                    <th class="text-end fw-bold">₹${formatNumber(totals['1_30'])}</th>
                    <th class="text-end fw-bold">₹${formatNumber(totals['31_60'])}</th>
                    <th class="text-end fw-bold">₹${formatNumber(totals['61_90'])}</th>
                    <th class="text-end fw-bold">₹${formatNumber(totals.over_90)}</th>
                    <th class="text-end fw-bold">₹${formatNumber(totals.total_due)}</th>
                    <th></th>
                </tr>
            `);

            // Initialize DataTable
            agingTable = $('#agingTable').DataTable({
                dom: 'Bfrtip',
                buttons: [
                    'copy', 'csv', 'excel', 'pdf', 'print', 'colvis'
                ],
                responsive: true,
                order: [[6, 'desc']], // Sort by total due descending
                columnDefs: [
                    { targets: [1, 2, 3, 4, 5, 6], className: 'text-end' },
                    { targets: [7], orderable: false }
                ]
            });
        }

        // View detailed aging for a vendor
        $(document).on('click', '.view-detailed-aging', function() {
            const vendorId = $(this).data('vendor-id');
            const vendorName = $(this).data('vendor-name');

            if (!currentAgingData) {
                Swal.fire('Error', 'No aging data available', 'error');
                return;
            }

            const vendorData = currentAgingData.data.find(item => item.vendor_id == vendorId);
            
            if (!vendorData || !vendorData.invoices) {
                // Load detailed data
                loadDetailedAgingData(vendorId, vendorName);
            } else {
                showDetailedModal(vendorData, vendorName);
            }
        });

        function loadDetailedAgingData(vendorId, vendorName) {
            const asOfDate = $('#aging-as-of-date').val();

            $.ajax({
                url: '{{ route('superadmin.purchase.v1.agingReport') }}',
                type: 'GET',
                data: {
                    as_of_date: asOfDate,
                    party_id: vendorId,
                    report_type: 'detailed'
                },
                success: function(response) {
                    if (response.success && response.data.length > 0) {
                        showDetailedModal(response.data[0], vendorName);
                    } else {
                        Swal.fire('Info', 'No detailed data available for this vendor', 'info');
                    }
                },
                error: function(xhr) {
                    Swal.fire('Error', 'Failed to load detailed data', 'error');
                }
            });
        }

        function showDetailedModal(vendorData, vendorName) {
            $('#modalVendorName').text(vendorName);
            
            const tbody = $('#detailedInvoicesList');
            tbody.empty();

            if (vendorData.invoices && vendorData.invoices.length > 0) {
                vendorData.invoices.forEach(function(invoice) {
                    const bucketClass = getBucketClass(invoice.bucket);
                    const bucketLabel = getBucketLabel(invoice.bucket);
                    
                    const tr = $(`
                        <tr class="${bucketClass}">
                            <td>${invoice.purchase_no}</td>
                            <td>${invoice.vendor_invoice_no || '-'}</td>
                            <td>${formatDate(invoice.purchase_date)}</td>
                            <td>${invoice.days_overdue} days</td>
                            <td class="text-end">₹${formatNumber(invoice.grand_total)}</td>
                            <td class="text-end">₹${formatNumber(invoice.paid_amount)}</td>
                            <td class="text-end fw-bold">₹${formatNumber(invoice.balance_amount)}</td>
                            <td><span class="badge bg-secondary">${bucketLabel}</span></td>
                        </tr>
                    `);
                    tbody.append(tr);
                });
            } else {
                tbody.html('<tr><td colspan="8" class="text-center">No detailed invoices available</td></tr>');
            }

            $('#detailedAgingModal').modal('show');
        }

        function getBucketClass(bucket) {
            const classes = {
                'current': 'bucket-current',
                '1_30': 'bucket-1-30',
                '31_60': 'bucket-31-60',
                '61_90': 'bucket-61-90',
                'over_90': 'bucket-over-90'
            };
            return classes[bucket] || '';
        }

        function getBucketLabel(bucket) {
            const labels = {
                'current': 'Current',
                '1_30': '1-30 Days',
                '31_60': '31-60 Days',
                '61_90': '61-90 Days',
                'over_90': '>90 Days'
            };
            return labels[bucket] || bucket;
        }

        // Event handlers
        $(document).on('click', '#applyAgingFilters, #refreshAgingReport', function() {
            loadAgingReport();
        });

        $(document).on('click', '#resetAgingFilters', function() {
            $('#aging-as-of-date').val('{{ date('Y-m-d') }}');
            $('#aging-party').val('');
            $('#aging-report-type').val('summary');
            loadAgingReport();
        });

        // View toggle buttons
        $(document).on('click', '#summaryViewBtn', function() {
            $('#aging-report-type').val('summary');
            $('#summaryViewBtn').addClass('active');
            $('#detailedViewBtn').removeClass('active');
            loadAgingReport();
        });

        $(document).on('click', '#detailedViewBtn', function() {
            $('#aging-report-type').val('detailed');
            $('#detailedViewBtn').addClass('active');
            $('#summaryViewBtn').removeClass('active');
            loadAgingReport();
        });

        // Export functionality
        $(document).on('click', '.export-aging', function(e) {
            e.preventDefault();
            const format = $(this).data('format');
            exportAgingReport(format);
        });

        function exportAgingReport(format) {
            const asOfDate = $('#aging-as-of-date').val();
            const partyId = $('#aging-party').val();
            const reportType = $('#aging-report-type').val();

            const params = new URLSearchParams({
                format: format,
                as_of_date: asOfDate,
                report_type: reportType
            });

            if (partyId) {
                params.append('party_id', partyId);
            }

            window.open('{{ route('superadmin.purchase.v1.agingExport') }}?' + params.toString(), '_blank');
        }

        // Utility functions
        function formatNumber(num) {
            return parseFloat(num).toLocaleString('en-IN', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            });
        }

        function formatDate(dateString) {
            return new Date(dateString).toLocaleDateString('en-IN');
        }

        // Vendor Payments functionality
        let paymentTable;

        // Initialize vendor payments tab when clicked
        $(document).on('click', '#pills-payment-tab', function() {
            if (!paymentTable) {
                initializePaymentTable();
            }
        });

        function initializePaymentTable() {
            // Initialize date range picker
            if ($.fn.daterangepicker) {
                $('#payment-date-range').daterangepicker({
                    autoUpdateInput: false,
                    locale: {
                        cancelLabel: 'Clear',
                        format: 'YYYY-MM-DD'
                    }
                }).on('apply.daterangepicker', function (ev, picker) {
                    $(this).val(picker.startDate.format('YYYY-MM-DD') + ' - ' + picker.endDate.format('YYYY-MM-DD'));
                }).on('cancel.daterangepicker', function () {
                    $(this).val('');
                });
            }

            // Initialize DataTable
            paymentTable = $('#paymentTable').DataTable({
                processing: true,
                serverSide: false,
                ajax: {
                    url: '{{ route("superadmin.purchase.v1.vendorPayments") }}',
                    type: 'GET',
                    data: function(d) {
                        return {
                            vendor_id: $('#payment-party').val(),
                            date_range: $('#payment-date-range').val(),
                            min_amount: $('#min-amount').val(),
                            max_amount: $('#max-amount').val()
                        };
                    },
                    dataSrc: function(json) {
                        if (json.success) {
                            return json.data;
                        } else {
                            console.error('Error loading vendor payments:', json.message);
                            return [];
                        }
                    },
                    error: function(xhr, error, thrown) {
                        console.error('AJAX Error:', error, thrown);
                        Swal.fire('Error', 'Failed to load vendor payments data', 'error');
                    }
                },
                columns: [
                    { data: 'payment_id' },
                    { data: 'purchase_id' },
                    { 
                        data: 'invoice_date',
                        render: function(data) {
                            return data ? new Date(data).toLocaleDateString('en-IN') : '-';
                        }
                    },
                    { data: 'party' },
                    { data: 'invoice_id' },
                    { 
                        data: 'grand_total',
                        render: function(data) {
                            return '₹' + parseFloat(data || 0).toFixed(2);
                        }
                    },
                    { 
                        data: 'amount',
                        render: function(data) {
                            return '₹' + parseFloat(data || 0).toFixed(2);
                        }
                    },
                    { 
                        data: 'balance_amount',
                        render: function(data) {
                            return '₹' + parseFloat(data || 0).toFixed(2);
                        }
                    },
                    { 
                        data: 'payment_status',
                        render: function(data) {
                            let badgeClass = 'badge-secondary';
                            if (data === 'paid') badgeClass = 'badge-success';
                            else if (data === 'partial') badgeClass = 'badge-warning';
                            else if (data === 'unpaid') badgeClass = 'badge-danger';
                            return '<span class="badge ' + badgeClass + '">' + data.charAt(0).toUpperCase() + data.slice(1) + '</span>';
                        }
                    },
                    { data: 'payment_mode' },
                    { data: 'reference' },
                    {
                        data: 'id',
                        orderable: false,
                        render: function(data) {
                            return `
                                <div class="btn-group" role="group">
                                    <a href="{{ url('superadmin/purchase/show') }}/${data}" class="btn btn-sm btn-outline-info" title="View">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                    <button type="button" class="btn btn-sm btn-outline-primary edit-payment" data-id="${data}" title="Edit Payment">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                </div>
                            `;
                        }
                    }
                ],
                dom: 'Bfrtip',
                buttons: [
                    'copy', 'csv', 'excel', 'pdf', 'print', 'colvis'
                ],
                responsive: true,
                order: [[1, 'desc']]
            });
        }

        // Apply payment filters
        $(document).on('click', '#applyPaymentFilters', function() {
            if (paymentTable) {
                paymentTable.ajax.reload();
            }
        });

        // Reset payment filters
        $(document).on('click', '#resetPaymentFilters', function() {
            $('#payment-party').val('');
            $('#payment-date-range').val('');
            $('#min-amount').val('');
            $('#max-amount').val('');
            if (paymentTable) {
                paymentTable.ajax.reload();
            }
        });

        // Add payment button
        $(document).on('click', '#addPaymentBtn', function() {
            $('#paymentForm')[0].reset();
            $('#paymentModalLabel').text('Add Payment');
            $('#payment_date').val(new Date().toISOString().split('T')[0]);
            loadUnpaidInvoices();
            $('#paymentModal').modal('show');
        });

        // Edit payment button
        $(document).on('click', '.edit-payment', function() {
            let purchaseId = $(this).data('id');
            // Load payment details for editing
            // This would require additional backend methods
            $('#paymentModalLabel').text('Edit Payment');
            $('#paymentModal').modal('show');
        });

        // Load unpaid invoices for payment form
        function loadUnpaidInvoices() {
            // This would load invoices with pending balances
            $('#payment_purchase_id').html('<option value="">Loading...</option>');
            
            // For now, we'll use a simple approach
            // In a full implementation, you'd create a separate endpoint
            $('#payment_purchase_id').html('<option value="">Select Purchase Invoice</option>');
        }

        // Submit payment form
        $(document).on('submit', '#paymentForm', function(e) {
            e.preventDefault();
            
            // This would require additional backend methods to handle payment updates
            Swal.fire('Info', 'Payment functionality will be implemented with additional backend methods', 'info');
        });

        // Function to reload payment table (called from index.blade.php)
        window.loadPaymentTableAgain = function() {
            if (paymentTable) {
                paymentTable.ajax.reload();
            } else {
                initializePaymentTable();
            }
        };

    });
</script>
@endpush
