@extends('layouts.dashboard.superadminapp')

@section('title', 'Edit Quotation')

@section('styles')
<style>
    .section-card {
        border: none;
        border-radius: 0.5rem;
        box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.1);
        margin-bottom: 1.5rem;
    }
    .section-card .card-header {
        background: linear-gradient(135deg, #4e73df 0%, #224abe 100%);
        color: #fff;
        font-weight: 600;
        padding: 0.75rem 1rem;
        border-radius: 0.5rem 0.5rem 0 0 !important;
        font-size: 0.9rem;
    }
    .section-card .card-header.bg-success-gradient {
        background: linear-gradient(135deg, #1cc88a 0%, #13855c 100%);
    }
    .section-card .card-header.bg-info-gradient {
        background: linear-gradient(135deg, #36b9cc 0%, #258391 100%);
    }
    .section-card .card-header.bg-warning-gradient {
        background: linear-gradient(135deg, #f6c23e 0%, #dda20a 100%);
    }
    .form-group label {
        font-weight: 600;
        font-size: 0.85rem;
        color: #5a5c69;
        margin-bottom: 0.25rem;
    }
    .form-control, .form-control-sm {
        font-size: 0.85rem;
    }
    .required {
        color: #e74a3b;
    }
    .product-table {
        width: 100%;
        font-size: 0.8rem;
    }
    .product-table th {
        background-color: #f8f9fc;
        font-weight: 600;
        font-size: 0.75rem;
        text-align: center;
        padding: 0.5rem;
        white-space: nowrap;
    }
    .product-table td {
        padding: 0.35rem;
        vertical-align: middle;
    }
    .product-table input, .product-table select {
        font-size: 0.8rem;
        padding: 0.25rem 0.5rem;
    }
    .product-table .btn-sm {
        padding: 0.2rem 0.4rem;
        font-size: 0.7rem;
    }
    .hidden {
        display: none;
    }
    .totals-section {
        background: #f8f9fc;
        border-radius: 0.5rem;
        padding: 1rem;
    }
    .totals-section .form-group {
        margin-bottom: 0.5rem;
    }
    .totals-section label {
        font-size: 0.8rem;
    }
    .totals-section input {
        text-align: right;
        font-weight: 600;
    }
    .checkbox-section {
        background: #f8f9fc;
        padding: 0.75rem 1rem;
        border-radius: 0.35rem;
        margin-bottom: 1rem;
    }
    .checkbox-section label {
        margin-bottom: 0;
        cursor: pointer;
    }
    .add-row-btn {
        margin-top: 0.5rem;
    }
</style>
@endsection

@section('content')
    <!-- Page Heading -->
    <div class="bg-white p-2 d-sm-flex align-items-center justify-content-between mb-4">
        <h1 class="h5 mb-0 text-gray-800">UEPL | Edit Quotation</h1>
        <div class="display-date">
            <span id="day">day</span>,
            <span id="daynum">00</span>
            <span id="month">month</span>
            <span id="year">0000</span>
        </div>
        <div class="display-time"></div>
    </div>

    <!-- Alerts -->
    @if (session('success'))
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fas fa-check-circle mr-2"></i>{{ session('success') }}
            <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                <span aria-hidden="true">&times;</span>
            </button>
        </div>
    @endif
    @if (session('error'))
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-circle mr-2"></i>{{ session('error') }}
            <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                <span aria-hidden="true">&times;</span>
            </button>
        </div>
    @endif
    @if ($errors->any())
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-triangle mr-2"></i>
            <strong>Please fix the following errors:</strong>
            <ul class="mb-0 mt-2">
                @foreach ($errors->all() as $error)
                    <li>{{ $error }}</li>
                @endforeach
            </ul>
            <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                <span aria-hidden="true">&times;</span>
            </button>
        </div>
    @endif

    <form id="quotationForm" method="POST" action="{{ route('superadmin.salesbom.quotation.update', $quotation->id) }}">
        @csrf

        <!-- Quotation & RFQ Information -->
        <div class="card section-card">
            <div class="card-header">
                <i class="fas fa-file-invoice mr-2"></i> Quotation & RFQ Information
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="quotationNo">Quotation No <span class="required">*</span></label>
                            <input type="text" class="form-control" id="quotationNo" name="quotation_no" value="{{ $quotation->quotation_no }}" readonly required>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="quotationDate">Quotation Date <span class="required">*</span></label>
                            <input type="date" class="form-control" id="quotationDate" name="quotation_date" value="{{ $quotation->quotation_date }}" required>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="rfqNo">RFQ No</label>
                            <select class="form-control selectpicker" id="rfqNo" name="rfq_no" data-live-search="true">
                                <option value="">Select RFQ No</option>
                                @foreach ($unassignedRfqs as $rfq)
                                    <option value="{{ $rfq->id }}" {{ $quotation->rfq_no == $rfq->id ? 'selected' : '' }}>{{ $rfq->rfq_no }}</option>
                                @endforeach
                                @if ($quotation->rfq_no && !$unassignedRfqs->contains('id', $quotation->rfq_no))
                                    @php $currentRfq = \App\Models\Rfq::find($quotation->rfq_no); @endphp
                                    @if($currentRfq)
                                        <option value="{{ $currentRfq->id }}" selected>{{ $currentRfq->rfq_no }}</option>
                                    @endif
                                @endif
                            </select>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="rfqDate">RFQ Date</label>
                            <input type="date" class="form-control" id="rfqDate" name="rfq_date" value="{{ $quotation->rfq_date }}">
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Customer Information -->
        <div class="card section-card">
            <div class="card-header bg-success-gradient">
                <i class="fas fa-user mr-2"></i> Customer Information
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="customerName">Customer Name</label>
                            <input type="text" class="form-control" id="customerName" value="{{ $customer->company ?? '' }}" readonly>
                            <input type="hidden" id="customerId" name="customer_name" value="{{ $quotation->customer_name }}">
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="custRefNo">Customer Reference No</label>
                            <input type="text" class="form-control" id="custRefNo" name="cust_ref_no" value="{{ $quotation->cust_ref_no }}" placeholder="Reference No">
                        </div>
                    </div>
                </div>
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="address">Address</label>
                            <textarea class="form-control" id="address" name="address" rows="2" placeholder="Customer Address">{{ $quotation->address }}</textarea>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="gst">GST No</label>
                            <input type="text" class="form-control" id="gst" name="gst" value="{{ $quotation->gst }}" placeholder="GST Number">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="phone">Contact Number</label>
                            <input type="tel" class="form-control" id="phone" name="phone" value="{{ $quotation->phone }}" placeholder="10-digit number" maxlength="10">
                        </div>
                    </div>
                </div>
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="email">Email</label>
                            <input type="email" class="form-control" id="email" name="email" value="{{ $quotation->email }}" placeholder="Email Address">
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Product Information -->
        <div class="card section-card">
            <div class="card-header bg-info-gradient">
                <i class="fas fa-box mr-2"></i> Product Information
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="productType">Product Type <span class="required">*</span></label>
                            <select class="form-control selectpicker" id="productType" name="product_type" data-live-search="true" required>
                                <option value="">Select Product</option>
                                @foreach($productTypes as $type)
                                    <option value="{{ $type->name }}" {{ $quotation->product_type == $type->name ? 'selected' : '' }}>{{ $type->name }}</option>
                                @endforeach
                                @if($quotation->product_type && !$productTypes->contains('name', $quotation->product_type))
                                    <option value="{{ $quotation->product_type }}" selected>{{ $quotation->product_type }}</option>
                                @endif
                            </select>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="partNo">Part No</label>
                            <input type="text" class="form-control" id="partNo" name="part_no" value="{{ $quotation->part_no }}" placeholder="Part No">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="project-materialNo">Project/Material No</label>
                            <input type="text" class="form-control" id="project-materialNo" name="project_material_no" value="{{ $quotation->project_material_no }}" placeholder="Project Material No">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="drawingNo">Drawing No</label>
                            <input type="text" class="form-control" id="drawingNo" name="drawing_no" value="{{ $quotation->drawing_no }}" placeholder="Drawing No">
                        </div>
                    </div>
                </div>
                <div class="row">
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="drawingRev">Drawing Revision</label>
                            <input type="text" class="form-control" id="drawingRev" name="drawing_rev" value="{{ $quotation->drawing_rev }}" placeholder="Revision">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="quantity">Quantity</label>
                            <input type="number" class="form-control" id="quantity" name="quantity" value="{{ $quotation->quantity }}" placeholder="Quantity">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="quantityUOM">Unit of Measurement</label>
                            <input type="text" class="form-control" id="quantityUOM" name="quantity_uom" value="{{ $quotation->quantity_uom }}" placeholder="UOM">
                        </div>
                    </div>
                </div>
                <div class="row">
                    <div class="col-md-12">
                        <div class="form-group">
                            <label for="description">Description</label>
                            <textarea class="form-control" id="description" name="description" rows="2" placeholder="Product Description">{{ $quotation->description }}</textarea>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Cost Tables Section -->
        <div class="card section-card">
            <div class="card-header bg-warning-gradient">
                <i class="fas fa-calculator mr-2"></i> Cost Breakdown
            </div>
            <div class="card-body">
                @php
                    $rawMaterials = is_array($quotation->raw_materials) ? $quotation->raw_materials : json_decode($quotation->raw_materials, true) ?? [];
                    $conversionCostsData = is_array($quotation->conversion_costs) ? $quotation->conversion_costs : json_decode($quotation->conversion_costs, true) ?? [];
                    $packingCharges = is_array($quotation->packing_charges) ? $quotation->packing_charges : json_decode($quotation->packing_charges, true) ?? [];
                @endphp

                <!-- Raw Material Charges -->
                <div class="checkbox-section">
                    <div class="custom-control custom-checkbox">
                        <input type="checkbox" class="custom-control-input" id="checkbox1" onclick="toggleTable(1)" {{ count($rawMaterials) > 0 ? 'checked' : '' }}>
                        <label class="custom-control-label font-weight-bold" for="checkbox1">
                            <i class="fas fa-cubes mr-1"></i> Raw Material Charges
                        </label>
                    </div>
                </div>
                <div class="table-responsive mb-3">
                    <table id="table1" class="table table-bordered product-table {{ count($rawMaterials) == 0 ? 'hidden' : '' }}">
                        <thead>
                            <tr>
                                <th style="width:3%">#</th>
                                <th style="width:12%">Raw Material</th>
                                <th style="width:7%">Rate(INR/kg)</th>
                                <th style="width:7%">Scrap Rate</th>
                                <th style="width:7%">Finish Wt</th>
                                <th style="width:7%">RM Type</th>
                                <th style="width:5%">Qty</th>
                                <th style="width:7%">Size 1</th>
                                <th style="width:7%">Size 2</th>
                                <th style="width:7%">Size 3</th>
                                <th style="width:7%">RM Weight</th>
                                <th style="width:8%">RM Cost(INR)</th>
                                <th style="width:8%">Value(INR)</th>
                                <th style="width:4%"></th>
                            </tr>
                        </thead>
                        <tbody id="rm-tbody">
                            @forelse ($rawMaterials as $i => $rm)
                            <tr>
                                <td class="text-center">{{ $i + 1 }}</td>
                                <td>
                                    <select name="raw_material[{{ $i }}][material_id]" class="form-control form-control-sm rm-material">
                                        <option value="">Select</option>
                                        @foreach ($allRawMaterials as $rawMaterial)
                                            <option value="{{ $rawMaterial->id }}" data-density="{{ $rawMaterial->density }}"
                                                {{ (($rm['material_id'] ?? $rm['material'] ?? '') == $rawMaterial->id) ? 'selected' : '' }}>
                                                {{ $rawMaterial->rm_grade }}
                                            </option>
                                        @endforeach
                                    </select>
                                    <input type="hidden" name="raw_material[{{ $i }}][density]" value="{{ $rm['density'] ?? '' }}">
                                </td>
                                <td><input type="number" name="raw_material[{{ $i }}][rate]" class="form-control form-control-sm rm-rate" step="0.01" value="{{ $rm['rate'] ?? '' }}"></td>
                                <td><input type="number" name="raw_material[{{ $i }}][scrap_rate]" class="form-control form-control-sm rm-scrap-rate" step="0.01" value="{{ $rm['scrap_rate'] ?? '' }}"></td>
                                <td><input type="number" name="raw_material[{{ $i }}][finish_weight]" class="form-control form-control-sm rm-finish-weight" step="0.01" value="{{ $rm['finish_weight'] ?? '' }}"></td>
                                <td>
                                    <select name="raw_material[{{ $i }}][type]" class="form-control form-control-sm rm-type">
                                        <option value="">Type</option>
                                        <option value="Rod" {{ ($rm['type'] ?? '') == 'Rod' ? 'selected' : '' }}>Rod</option>
                                        <option value="Plate" {{ ($rm['type'] ?? '') == 'Plate' ? 'selected' : '' }}>Plate</option>
                                        <option value="Pipe" {{ ($rm['type'] ?? '') == 'Pipe' ? 'selected' : '' }}>Pipe</option>
                                        <option value="Other" {{ ($rm['type'] ?? '') == 'Other' ? 'selected' : '' }}>Other</option>
                                    </select>
                                </td>
                                <td><input type="number" name="raw_material[{{ $i }}][quantity]" class="form-control form-control-sm rm-quantity" min="1" value="{{ $rm['quantity'] ?? '' }}"></td>
                                <td><input type="number" name="raw_material[{{ $i }}][size1]" class="form-control form-control-sm rm-size1" value="{{ $rm['size1'] ?? '' }}"></td>
                                <td><input type="number" name="raw_material[{{ $i }}][size2]" class="form-control form-control-sm rm-size2" value="{{ $rm['size2'] ?? '' }}"></td>
                                <td><input type="number" name="raw_material[{{ $i }}][size3]" class="form-control form-control-sm rm-size3" value="{{ $rm['size3'] ?? '' }}"></td>
                                <td><input type="text" name="raw_material[{{ $i }}][rm_weight]" class="form-control form-control-sm rm-weight" readonly value="{{ $rm['rm_weight'] ?? '' }}"></td>
                                <td><input type="number" name="raw_material[{{ $i }}][rm_cost]" class="form-control form-control-sm rm-cost" step="0.01" value="{{ $rm['rm_cost'] ?? '' }}"></td>
                                <td><input type="text" name="raw_material[{{ $i }}][value]" class="form-control form-control-sm rm-value" readonly value="{{ $rm['value'] ?? '' }}"></td>
                                <td class="text-center"><button type="button" class="btn btn-danger btn-sm" onclick="deleteRow(this)"><i class="fa fa-trash"></i></button></td>
                            </tr>
                            @empty
                            <tr>
                                <td class="text-center">1</td>
                                <td>
                                    <select name="raw_material[0][material_id]" class="form-control form-control-sm rm-material">
                                        <option value="">Select</option>
                                        @foreach ($allRawMaterials as $rawMaterial)
                                            <option value="{{ $rawMaterial->id }}" data-density="{{ $rawMaterial->density }}">{{ $rawMaterial->rm_grade }}</option>
                                        @endforeach
                                    </select>
                                    <input type="hidden" name="raw_material[0][density]">
                                </td>
                                <td><input type="number" name="raw_material[0][rate]" class="form-control form-control-sm rm-rate" step="0.01"></td>
                                <td><input type="number" name="raw_material[0][scrap_rate]" class="form-control form-control-sm rm-scrap-rate" step="0.01"></td>
                                <td><input type="number" name="raw_material[0][finish_weight]" class="form-control form-control-sm rm-finish-weight" step="0.01"></td>
                                <td>
                                    <select name="raw_material[0][type]" class="form-control form-control-sm rm-type">
                                        <option value="">Type</option>
                                        <option value="Rod">Rod</option>
                                        <option value="Plate">Plate</option>
                                        <option value="Pipe">Pipe</option>
                                        <option value="Other">Other</option>
                                    </select>
                                </td>
                                <td><input type="number" name="raw_material[0][quantity]" class="form-control form-control-sm rm-quantity" min="1"></td>
                                <td><input type="number" name="raw_material[0][size1]" class="form-control form-control-sm rm-size1"></td>
                                <td><input type="number" name="raw_material[0][size2]" class="form-control form-control-sm rm-size2"></td>
                                <td><input type="number" name="raw_material[0][size3]" class="form-control form-control-sm rm-size3"></td>
                                <td><input type="text" name="raw_material[0][rm_weight]" class="form-control form-control-sm rm-weight" readonly></td>
                                <td><input type="number" name="raw_material[0][rm_cost]" class="form-control form-control-sm rm-cost" step="0.01"></td>
                                <td><input type="text" name="raw_material[0][value]" class="form-control form-control-sm rm-value" readonly></td>
                                <td class="text-center"><button type="button" class="btn btn-danger btn-sm" onclick="deleteRow(this)"><i class="fa fa-trash"></i></button></td>
                            </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
                <div id="add-row-1" class="add-row-btn" style="{{ count($rawMaterials) == 0 ? 'display: none;' : '' }}">
                    <button type="button" class="btn btn-outline-primary btn-sm" id="addRowRM">
                        <i class="fas fa-plus mr-1"></i> Add Raw Material Row
                    </button>
                </div>

                <hr class="my-4">

                <!-- Conversion Cost -->
                <div class="checkbox-section">
                    <div class="custom-control custom-checkbox">
                        <input type="checkbox" class="custom-control-input" id="checkbox2" onclick="toggleTable(2)" {{ count($conversionCostsData) > 0 ? 'checked' : '' }}>
                        <label class="custom-control-label font-weight-bold" for="checkbox2">
                            <i class="fas fa-cogs mr-1"></i> Conversion Cost
                        </label>
                    </div>
                </div>
                <div class="table-responsive mb-3">
                    <table id="table2" class="table table-bordered product-table {{ count($conversionCostsData) == 0 ? 'hidden' : '' }}">
                        <thead>
                            <tr>
                                <th style="width:5%">#</th>
                                <th style="width:30%">Description</th>
                                <th style="width:15%">Quantity</th>
                                <th style="width:15%">Unit</th>
                                <th style="width:15%">Rate(INR)</th>
                                <th style="width:15%">Value</th>
                                <th style="width:5%"></th>
                            </tr>
                        </thead>
                        <tbody id="cc-tbody">
                            @forelse ($conversionCostsData as $i => $cc)
                            <tr>
                                <td class="text-center">{{ $i + 1 }}</td>
                                <td>
                                    <input list="conversionCost-options" name="conversion_cost[{{ $i }}][desc]" class="form-control form-control-sm" placeholder="Select process" value="{{ $cc['desc'] ?? '' }}">
                                    <datalist id="conversionCost-options">
                                        @foreach ($conversionCosts as $cost)
                                            <option value="{{ $cost }}"></option>
                                        @endforeach
                                    </datalist>
                                </td>
                                <td><input type="number" name="conversion_cost[{{ $i }}][quantity]" class="form-control form-control-sm cc-quantity" min="1" value="{{ $cc['quantity'] ?? '' }}"></td>
                                <td><input type="text" name="conversion_cost[{{ $i }}][unit]" class="form-control form-control-sm" placeholder="Unit" value="{{ $cc['unit'] ?? '' }}"></td>
                                <td><input type="number" name="conversion_cost[{{ $i }}][rate]" class="form-control form-control-sm cc-rate" min="0" step="0.01" value="{{ $cc['rate'] ?? '' }}"></td>
                                <td><input type="text" name="conversion_cost[{{ $i }}][value]" class="form-control form-control-sm cc-value" readonly value="{{ $cc['value'] ?? '' }}"></td>
                                <td class="text-center"><button type="button" class="btn btn-danger btn-sm" onclick="deleteRow(this)"><i class="fa fa-trash"></i></button></td>
                            </tr>
                            @empty
                            <tr>
                                <td class="text-center">1</td>
                                <td>
                                    <input list="conversionCost-options" name="conversion_cost[0][desc]" class="form-control form-control-sm" placeholder="Select process">
                                    <datalist id="conversionCost-options">
                                        @foreach ($conversionCosts as $cost)
                                            <option value="{{ $cost }}"></option>
                                        @endforeach
                                    </datalist>
                                </td>
                                <td><input type="number" name="conversion_cost[0][quantity]" class="form-control form-control-sm cc-quantity" min="1"></td>
                                <td><input type="text" name="conversion_cost[0][unit]" class="form-control form-control-sm" placeholder="Unit"></td>
                                <td><input type="number" name="conversion_cost[0][rate]" class="form-control form-control-sm cc-rate" min="0" step="0.01"></td>
                                <td><input type="text" name="conversion_cost[0][value]" class="form-control form-control-sm cc-value" readonly></td>
                                <td class="text-center"><button type="button" class="btn btn-danger btn-sm" onclick="deleteRow(this)"><i class="fa fa-trash"></i></button></td>
                            </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
                <div id="add-row-2" class="add-row-btn" style="{{ count($conversionCostsData) == 0 ? 'display: none;' : '' }}">
                    <button type="button" class="btn btn-outline-primary btn-sm" id="addRowCC">
                        <i class="fas fa-plus mr-1"></i> Add Conversion Cost Row
                    </button>
                </div>

                <hr class="my-4">

                <!-- Packing Charges -->
                <div class="checkbox-section">
                    <div class="custom-control custom-checkbox">
                        <input type="checkbox" class="custom-control-input" id="checkbox3" onclick="toggleTable(3)" {{ count($packingCharges) > 0 ? 'checked' : '' }}>
                        <label class="custom-control-label font-weight-bold" for="checkbox3">
                            <i class="fas fa-box-open mr-1"></i> Packing Charges
                        </label>
                    </div>
                </div>
                <div class="table-responsive mb-3">
                    <table id="table3" class="table table-bordered product-table {{ count($packingCharges) == 0 ? 'hidden' : '' }}">
                        <thead>
                            <tr>
                                <th style="width:5%">#</th>
                                <th style="width:30%">Description</th>
                                <th style="width:15%">Quantity</th>
                                <th style="width:15%">Unit</th>
                                <th style="width:15%">Rate(INR)</th>
                                <th style="width:15%">Value</th>
                                <th style="width:5%"></th>
                            </tr>
                        </thead>
                        <tbody id="pc-tbody">
                            @forelse ($packingCharges as $i => $pc)
                            <tr>
                                <td class="text-center">{{ $i + 1 }}</td>
                                <td><input type="text" name="packing[{{ $i }}][desc]" class="form-control form-control-sm" value="{{ $pc['desc'] ?? 'Packing cost' }}"></td>
                                <td><input type="number" name="packing[{{ $i }}][quantity]" class="form-control form-control-sm pc-quantity" min="1" value="{{ $pc['quantity'] ?? '' }}"></td>
                                <td><input type="text" name="packing[{{ $i }}][unit]" class="form-control form-control-sm" placeholder="Unit" value="{{ $pc['unit'] ?? '' }}"></td>
                                <td><input type="number" name="packing[{{ $i }}][rate]" class="form-control form-control-sm pc-rate" min="0" step="0.01" value="{{ $pc['rate'] ?? '' }}"></td>
                                <td><input type="text" name="packing[{{ $i }}][value]" class="form-control form-control-sm pc-value" readonly value="{{ $pc['value'] ?? '' }}"></td>
                                <td class="text-center"><button type="button" class="btn btn-danger btn-sm" onclick="deleteRow(this)"><i class="fa fa-trash"></i></button></td>
                            </tr>
                            @empty
                            <tr>
                                <td class="text-center">1</td>
                                <td><input type="text" name="packing[0][desc]" class="form-control form-control-sm" value="Packing cost"></td>
                                <td><input type="number" name="packing[0][quantity]" class="form-control form-control-sm pc-quantity" min="1"></td>
                                <td><input type="text" name="packing[0][unit]" class="form-control form-control-sm" placeholder="Unit"></td>
                                <td><input type="number" name="packing[0][rate]" class="form-control form-control-sm pc-rate" min="0" step="0.01"></td>
                                <td><input type="text" name="packing[0][value]" class="form-control form-control-sm pc-value" readonly></td>
                                <td class="text-center"><button type="button" class="btn btn-danger btn-sm" onclick="deleteRow(this)"><i class="fa fa-trash"></i></button></td>
                            </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
                <div id="add-row-3" class="add-row-btn" style="{{ count($packingCharges) == 0 ? 'display: none;' : '' }}">
                    <button type="button" class="btn btn-outline-primary btn-sm" id="addRowPC">
                        <i class="fas fa-plus mr-1"></i> Add Packing Charge Row
                    </button>
                </div>
            </div>
        </div>

        <!-- Totals & Note Section -->
        <div class="card section-card">
            <div class="card-header">
                <i class="fas fa-rupee-sign mr-2"></i> Summary & Totals
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="quoteNote">Note / Remarks</label>
                            <textarea name="note" id="quoteNote" class="form-control" rows="4" placeholder="Additional notes or remarks">{{ $quotation->note }}</textarea>
                        </div>
                        <div class="form-group">
                            <label for="GT-in-letters">Grand Total in Words (INR)</label>
                            <input type="text" class="form-control" id="GT-in-letters" readonly>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="totals-section">
                            <div class="form-group row align-items-center">
                                <label class="col-md-4 col-form-label">Sub Total</label>
                                <div class="col-md-8">
                                    <input type="text" class="form-control" name="sub_total" id="totalAmount" value="{{ $quotation->sub_total }}" readonly>
                                </div>
                            </div>
                            <div class="form-group row align-items-center">
                                <label class="col-md-4 col-form-label">Discount (%)</label>
                                <div class="col-md-8">
                                    <input type="number" class="form-control" id="qt-discount" name="discount" value="{{ $quotation->discount }}" step="0.01" oninput="calculateTotals()">
                                </div>
                            </div>
                            <div class="form-group row align-items-center">
                                <label class="col-md-4 col-form-label">Round Up</label>
                                <div class="col-md-8">
                                    <input type="text" class="form-control" id="roundUp" name="round_up" value="{{ $quotation->round_up }}" readonly>
                                </div>
                            </div>
                            <hr>
                            <div class="form-group row align-items-center mb-0">
                                <label class="col-md-4 col-form-label font-weight-bold text-primary">Grand Total</label>
                                <div class="col-md-8">
                                    <input type="text" class="form-control form-control-lg font-weight-bold text-primary" id="grandTotal" name="grand_total" value="{{ $quotation->grand_total }}" readonly>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Action Buttons -->
        <div class="card section-card">
            <div class="card-body">
                <div class="d-flex justify-content-between">
                    <a href="{{ route('superadmin.salesbom.quotation.quotationdetails') }}" class="btn btn-secondary">
                        <i class="fas fa-arrow-left mr-1"></i> Back to List
                    </a>
                    <div>
                        <a href="{{ route('superadmin.salesbom.quotation.view', $quotation->id) }}" class="btn btn-info mr-2">
                            <i class="fas fa-eye mr-1"></i> View
                        </a>
                        <button type="submit" class="btn btn-success" id="submitBtn">
                            <span class="spinner-border spinner-border-sm d-none" role="status"></span>
                            <span class="btn-text"><i class="fas fa-save mr-1"></i> Update Quotation</span>
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </form>
@endsection

@section('scripts')
<script>
$(document).ready(function() {
    // Initialize selectpicker
    $('.selectpicker').selectpicker();

    // Phone number validation - only digits
    $('#phone').on('input', function() {
        this.value = this.value.replace(/[^0-9]/g, '').slice(0, 10);
    });

    // Form submit with loading spinner
    $('#quotationForm').on('submit', function(e) {
        var $btn = $('#submitBtn');
        var $spinner = $btn.find('.spinner-border');
        var $text = $btn.find('.btn-text');

        $btn.prop('disabled', true);
        $spinner.removeClass('d-none');
        $text.html('Updating...');
    });

    // Row counters
    let rmRow = $('#rm-tbody tr').length;
    let ccRow = $('#cc-tbody tr').length;
    let pcRow = $('#pc-tbody tr').length;

    // Add Row - Raw Material
    $('#addRowRM').click(function() {
        let idx = rmRow++;
        let options = '<option value="">Select</option>';
        @foreach($allRawMaterials as $rawMaterial)
        options += '<option value="{{ $rawMaterial->id }}" data-density="{{ $rawMaterial->density }}">{{ $rawMaterial->rm_grade }}</option>';
        @endforeach

        $('#rm-tbody').append(`
            <tr>
                <td class="text-center">${idx+1}</td>
                <td>
                    <select name="raw_material[${idx}][material_id]" class="form-control form-control-sm rm-material">
                        ${options}
                    </select>
                    <input type="hidden" name="raw_material[${idx}][density]">
                </td>
                <td><input type="number" name="raw_material[${idx}][rate]" class="form-control form-control-sm rm-rate" step="0.01"></td>
                <td><input type="number" name="raw_material[${idx}][scrap_rate]" class="form-control form-control-sm rm-scrap-rate" step="0.01"></td>
                <td><input type="number" name="raw_material[${idx}][finish_weight]" class="form-control form-control-sm rm-finish-weight" step="0.01"></td>
                <td>
                    <select name="raw_material[${idx}][type]" class="form-control form-control-sm rm-type">
                        <option value="">Type</option>
                        <option value="Rod">Rod</option>
                        <option value="Plate">Plate</option>
                        <option value="Pipe">Pipe</option>
                        <option value="Other">Other</option>
                    </select>
                </td>
                <td><input type="number" name="raw_material[${idx}][quantity]" class="form-control form-control-sm rm-quantity" min="1"></td>
                <td><input type="number" name="raw_material[${idx}][size1]" class="form-control form-control-sm rm-size1"></td>
                <td><input type="number" name="raw_material[${idx}][size2]" class="form-control form-control-sm rm-size2"></td>
                <td><input type="number" name="raw_material[${idx}][size3]" class="form-control form-control-sm rm-size3"></td>
                <td><input type="text" name="raw_material[${idx}][rm_weight]" class="form-control form-control-sm rm-weight" readonly></td>
                <td><input type="number" name="raw_material[${idx}][rm_cost]" class="form-control form-control-sm rm-cost" step="0.01"></td>
                <td><input type="text" name="raw_material[${idx}][value]" class="form-control form-control-sm rm-value" readonly></td>
                <td class="text-center"><button type="button" class="btn btn-danger btn-sm" onclick="deleteRow(this)"><i class="fa fa-trash"></i></button></td>
            </tr>
        `);
    });

    // Add Row - Conversion Cost
    $('#addRowCC').click(function() {
        let idx = ccRow++;
        $('#cc-tbody').append(`
            <tr>
                <td class="text-center">${idx+1}</td>
                <td><input list="conversionCost-options" name="conversion_cost[${idx}][desc]" class="form-control form-control-sm" placeholder="Select process"></td>
                <td><input type="number" name="conversion_cost[${idx}][quantity]" class="form-control form-control-sm cc-quantity" min="1"></td>
                <td><input type="text" name="conversion_cost[${idx}][unit]" class="form-control form-control-sm" placeholder="Unit"></td>
                <td><input type="number" name="conversion_cost[${idx}][rate]" class="form-control form-control-sm cc-rate" min="0" step="0.01"></td>
                <td><input type="text" name="conversion_cost[${idx}][value]" class="form-control form-control-sm cc-value" readonly></td>
                <td class="text-center"><button type="button" class="btn btn-danger btn-sm" onclick="deleteRow(this)"><i class="fa fa-trash"></i></button></td>
            </tr>
        `);
    });

    // Add Row - Packing Charges
    $('#addRowPC').click(function() {
        let idx = pcRow++;
        $('#pc-tbody').append(`
            <tr>
                <td class="text-center">${idx+1}</td>
                <td><input type="text" name="packing[${idx}][desc]" class="form-control form-control-sm" placeholder="Description"></td>
                <td><input type="number" name="packing[${idx}][quantity]" class="form-control form-control-sm pc-quantity" min="1"></td>
                <td><input type="text" name="packing[${idx}][unit]" class="form-control form-control-sm" placeholder="Unit"></td>
                <td><input type="number" name="packing[${idx}][rate]" class="form-control form-control-sm pc-rate" min="0" step="0.01"></td>
                <td><input type="text" name="packing[${idx}][value]" class="form-control form-control-sm pc-value" readonly></td>
                <td class="text-center"><button type="button" class="btn btn-danger btn-sm" onclick="deleteRow(this)"><i class="fa fa-trash"></i></button></td>
            </tr>
        `);
    });

    // Value calculations for tables
    $(document).on('input', '.rm-rate, .rm-quantity, .rm-cost', function() {
        let row = $(this).closest('tr');
        let quantity = parseFloat(row.find('.rm-quantity').val()) || 0;
        let cost = parseFloat(row.find('.rm-cost').val()) || 0;
        row.find('.rm-value').val((quantity * cost).toFixed(2));
        calculateTotals();
    });

    $(document).on('input', '.cc-rate, .cc-quantity', function() {
        let row = $(this).closest('tr');
        let rate = parseFloat(row.find('.cc-rate').val()) || 0;
        let quantity = parseFloat(row.find('.cc-quantity').val()) || 0;
        row.find('.cc-value').val((rate * quantity).toFixed(2));
        calculateTotals();
    });

    $(document).on('input', '.pc-rate, .pc-quantity', function() {
        let row = $(this).closest('tr');
        let rate = parseFloat(row.find('.pc-rate').val()) || 0;
        let quantity = parseFloat(row.find('.pc-quantity').val()) || 0;
        row.find('.pc-value').val((rate * quantity).toFixed(2));
        calculateTotals();
    });

    // Raw Material type change - update placeholders
    $(document).on('change', '.rm-type', function() {
        updateSizePlaceholders(this);
        calculateRMCost($(this).closest('tr')[0]);
    });

    // Raw Material selection - get density
    $(document).on('change', '.rm-material', function() {
        let density = $(this).find(':selected').data('density');
        $(this).closest('tr').find('input[name$="[density]"]').val(density);
        calculateRMCost($(this).closest('tr')[0]);
    });

    // Calculate RM cost on relevant input changes
    $(document).on('input', '.rm-rate, .rm-scrap-rate, .rm-finish-weight, .rm-size1, .rm-size2, .rm-size3', function() {
        calculateRMCost($(this).closest('tr')[0]);
    });

    // Initialize on page load
    $('#rm-tbody tr').each(function() {
        const typeSelect = $(this).find('.rm-type')[0];
        if (typeSelect) {
            updateSizePlaceholders(typeSelect);
            calculateRMCost(this);
        }
    });

    // Calculate totals on page load
    calculateTotals();
});

// Toggle table visibility
function toggleTable(num) {
    $("#table" + num).toggleClass('hidden');
    $("#add-row-" + num).toggle();
}

// Delete row
function deleteRow(btn) {
    $(btn).closest('tr').remove();
    calculateTotals();
}

// Calculate totals
function calculateTotals() {
    let subtotal = 0;
    $('.rm-value, .cc-value, .pc-value').each(function() {
        subtotal += parseFloat($(this).val()) || 0;
    });
    $('#totalAmount').val(subtotal.toFixed(2));

    let discount = parseFloat($('#qt-discount').val()) || 0;
    let afterDiscount = subtotal - (subtotal * discount / 100);
    let roundUp = Math.round(afterDiscount) - afterDiscount;
    $('#roundUp').val(roundUp.toFixed(2));

    let grandTotal = afterDiscount + roundUp;
    $('#grandTotal').val(grandTotal.toFixed(2));
    $('#GT-in-letters').val(numberToWords(Math.round(grandTotal)) + ' Rupees Only');
}

// Update size placeholders based on RM type
function updateSizePlaceholders(selectElement) {
    let row = $(selectElement).closest('tr');
    let size1 = row.find('.rm-size1')[0];
    let size2 = row.find('.rm-size2')[0];
    let size3 = row.find('.rm-size3')[0];
    let rmCost = row.find('.rm-cost')[0];
    let type = $(selectElement).val();

    if (type === 'Rod') {
        size1.placeholder = 'Diameter';
        size2.placeholder = 'Length';
        size3.placeholder = '';
        size1.readOnly = false;
        size2.readOnly = false;
        size3.readOnly = true;
        rmCost.readOnly = true;
    } else if (type === 'Plate') {
        size1.placeholder = 'Thickness';
        size2.placeholder = 'Length';
        size3.placeholder = 'Width';
        size1.readOnly = false;
        size2.readOnly = false;
        size3.readOnly = false;
        rmCost.readOnly = true;
    } else if (type === 'Pipe') {
        size1.placeholder = 'Outer Dia';
        size2.placeholder = 'Inner Dia';
        size3.placeholder = 'Length';
        size1.readOnly = false;
        size2.readOnly = false;
        size3.readOnly = false;
        rmCost.readOnly = true;
    } else {
        size1.placeholder = 'Size 1';
        size2.placeholder = 'Size 2';
        size3.placeholder = 'Size 3';
        size1.readOnly = true;
        size2.readOnly = true;
        size3.readOnly = true;
        rmCost.readOnly = false;
    }
}

// Calculate RM Cost
function calculateRMCost(rowElement) {
    let row = $(rowElement);
    let rmRate = parseFloat(row.find('.rm-rate').val()) || 0;
    let scrapRate = parseFloat(row.find('.rm-scrap-rate').val()) || 0;
    let finishWeight = parseFloat(row.find('.rm-finish-weight').val()) || 0;
    let size1 = parseFloat(row.find('.rm-size1').val()) || 0;
    let size2 = parseFloat(row.find('.rm-size2').val()) || 0;
    let size3 = parseFloat(row.find('.rm-size3').val()) || 0;
    let rmType = row.find('.rm-type').val();
    let density = parseFloat(row.find('input[name$="[density]"]').val()) || 0;

    let rmWeight = 0, rmCost = 0;

    if (rmType === 'Rod') {
        rmWeight = 3.142 * Math.pow(size1 / 2, 2) * size2 * density;
    } else if (rmType === 'Plate') {
        rmWeight = size1 * size2 * size3 * density;
    } else if (rmType === 'Pipe') {
        rmWeight = (3.142 * Math.pow(size1 / 2, 2) * size3 - 3.142 * Math.pow(size2 / 2, 2) * size3) * density;
    }

    if (rmWeight > 0) {
        let scrapWeight = rmWeight - finishWeight;
        rmCost = (rmWeight * rmRate) - (0.8 * (scrapWeight * scrapRate));
    }

    row.find('.rm-weight').val(rmWeight > 0 ? rmWeight.toFixed(2) : '');
    row.find('.rm-cost').val(rmCost > 0 ? rmCost.toFixed(2) : '');

    // Trigger value calculation
    let quantity = parseFloat(row.find('.rm-quantity').val()) || 0;
    row.find('.rm-value').val((quantity * rmCost).toFixed(2));
    calculateTotals();
}

// Number to words converter (Indian format)
function numberToWords(num) {
    if (num === 0) return 'Zero';

    const ones = ['', 'One', 'Two', 'Three', 'Four', 'Five', 'Six', 'Seven', 'Eight', 'Nine',
                  'Ten', 'Eleven', 'Twelve', 'Thirteen', 'Fourteen', 'Fifteen', 'Sixteen',
                  'Seventeen', 'Eighteen', 'Nineteen'];
    const tens = ['', '', 'Twenty', 'Thirty', 'Forty', 'Fifty', 'Sixty', 'Seventy', 'Eighty', 'Ninety'];

    function convertLessThanHundred(n) {
        if (n < 20) return ones[n];
        return tens[Math.floor(n / 10)] + (n % 10 ? ' ' + ones[n % 10] : '');
    }

    function convertLessThanThousand(n) {
        if (n < 100) return convertLessThanHundred(n);
        return ones[Math.floor(n / 100)] + ' Hundred' + (n % 100 ? ' ' + convertLessThanHundred(n % 100) : '');
    }

    if (num < 0) return 'Negative ' + numberToWords(-num);
    if (num < 1000) return convertLessThanThousand(num);
    if (num < 100000) {
        return convertLessThanHundred(Math.floor(num / 1000)) + ' Thousand' +
               (num % 1000 ? ' ' + convertLessThanThousand(num % 1000) : '');
    }
    if (num < 10000000) {
        return convertLessThanHundred(Math.floor(num / 100000)) + ' Lakh' +
               (num % 100000 ? ' ' + numberToWords(num % 100000) : '');
    }
    return convertLessThanHundred(Math.floor(num / 10000000)) + ' Crore' +
           (num % 10000000 ? ' ' + numberToWords(num % 10000000) : '');
}
</script>
@endsection
