@extends('layouts.dashboard.superadminapp')

@section('title', 'Manufacturing Plan Board')

@section('content')
@push('styles')
<style>
    /* Premium Page Styles */
    .page-header-gradient {
        background: linear-gradient(135deg, #4e73df 0%, #224abe 100%);
        padding: 2rem 2rem 3rem 2rem;
        border-radius: 0 0 20px 20px;
        margin-bottom: 2rem;
        box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        color: white;
        margin-left: -15px;
        margin-right: -15px;
    }
    
    .card-premium {
        border: none;
        border-radius: 15px;
        box-shadow: 0 5px 20px rgba(0,0,0,0.05);
        transition: transform 0.2s ease, box-shadow 0.2s ease;
        overflow: hidden;
    }
    
    .card-premium:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 25px rgba(0,0,0,0.08);
    }

    .info-label {
        font-size: 0.75rem;
        text-transform: uppercase;
        letter-spacing: 0.05em;
        color: #858796;
        font-weight: 700;
        margin-bottom: 0.25rem;
    }

    .info-value {
        font-size: 1rem;
        font-weight: 600;
        color: #3a3b45;
    }
    
    /* Timeline Process Card Styles */
    .timeline-container {
        position: relative;
        padding-left: 20px;
    }
    .timeline-container::before {
        content: '';
        position: absolute;
        left: 0;
        top: 10px;
        bottom: 10px;
        width: 2px;
        background: #e3e6f0;
        border-radius: 2px;
    }
    
    .process-card {
        border-left: 4px solid #4e73df;
        margin-bottom: 1.5rem;
        margin-left: 1rem;
    }
    
    .process-card::before {
        content: '';
        position: absolute;
        left: -29px;
        top: 20px;
        width: 16px;
        height: 16px;
        background: #fff;
        border: 4px solid #4e73df;
        border-radius: 50%;
        z-index: 1;
    }

    .badge-premium {
        padding: 0.5em 1em;
        border-radius: 30px;
        font-weight: 600;
        font-size: 0.75rem;
        letter-spacing: 0.5px;
    }

    /* Modal Form Enhancements */
    .modal-content {
        border: none;
        border-radius: 16px;
        box-shadow: 0 10px 40px rgba(0,0,0,0.2);
    }
    .modal-header {
        background: #f8f9fc;
        border-bottom: 1px solid #e3e6f0;
        border-radius: 16px 16px 0 0;
        padding: 1.5rem;
    }
    .modal-body {
        padding: 2rem;
    }
    .form-control {
        border-radius: 8px;
        height: calc(1.5em + 1rem + 2px);
        padding: 0.5rem 1rem;
        border-color: #d1d3e2;
    }
    .form-control:focus {
        border-color: #4e73df;
        box-shadow: 0 0 0 0.2rem rgba(78, 115, 223, 0.1);
    }
    .form-group label {
        font-weight: 600;
        color: #5a5c69;
        font-size: 0.85rem;
    }
</style>
@endpush

    <meta name="csrf-token" content="{{ csrf_token() }}">

    <!-- Header Section -->
    <div class="page-header-gradient">
        <div class="container-fluid">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h1 class="h3 font-weight-bold mb-1">Manufacturing Plan Board</h1>
                    <p class="mb-0 opacity-75 small">Manage Route Card Processes and Schedules</p>
                </div>
                <a href="{{ url('superadmin/manufacturing/planning') }}" class="btn btn-light btn-sm font-weight-bold shadow-sm text-primary">
                    <i class="fas fa-arrow-left mr-2"></i>Back to Planning
                </a>
            </div>
        </div>
    </div>

    <div class="container-fluid" style="margin-top: -2.5rem;">
        {{-- Flash Messages --}}
        @if(session('success'))
            <div class="alert alert-success border-0 shadow-sm alert-dismissible fade show mb-4" role="alert" style="border-radius: 10px;">
                <i class="fas fa-check-circle mr-2"></i> {{ session('success') }}
                <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
        @endif
        @if(session('error'))
            <div class="alert alert-danger border-0 shadow-sm alert-dismissible fade show mb-4" role="alert" style="border-radius: 10px;">
                 <i class="fas fa-exclamation-circle mr-2"></i> {{ session('error') }}
                <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
        @endif

        <!-- RC Header Placeholder -->
        <div id="rcHeader" class="mb-4">
            <!-- Loaded by JS -->
            <div class="card card-premium p-5 text-center">
                <div class="spinner-border text-primary" role="status"><span class="sr-only">Loading...</span></div>
            </div>
        </div>

        <!-- The Board (Timeline) -->
        <div class="timeline-container board">
             <!-- Processes Loaded by JS -->
        </div>
    </div>

    <!-- Edit/Add Modal -->
    <div class="modal fade" id="processModal" tabindex="-1" role="dialog" aria-labelledby="processModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg" role="document"> <!-- modal-lg for better spacing -->
            <form id="processForm" class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title font-weight-bold text-gray-800" id="processModalLabel">
                        <i class="fas fa-tasks mr-2 text-primary"></i>Edit Process
                    </h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <div id="processMessage" class="alert" style="display:none"></div>
                    <input type="hidden" id="process_id">
                    
                    <div class="row">
                        <div class="form-group col-md-12">
                            <label for="process_name">Process Name <span class="text-danger">*</span></label>
                            <select class="form-control" id="process_name" required>
                                <!-- Populated by JS -->
                            </select>
                        </div>
                        <div class="form-group col-md-12">
                            <label for="description">Description</label>
                            <textarea class="form-control" id="description" rows="2" placeholder="Process details..."></textarea>
                        </div>
                    </div>
                     <hr class="my-3">
                    <div class="row">
                        <div class="form-group col-md-6">
                            <label for="machine">Machine</label>
                            <select class="form-control" id="machine">
                                <!-- Populated by JS -->
                            </select>
                        </div>
                        <div class="form-group col-md-6">
                            <label for="operator">Operator <span class="text-danger">*</span></label>
                            <select class="form-control" id="operator" required>
                                <option value="">Select Operator</option>
                                @foreach ($operators as $oid => $oname)
                                    <option value="{{ $oid }}">{{ $oname }}</option>
                                @endforeach
                            </select>
                        </div>
                    </div>
                    <div class="row">
                        <div class="form-group col-md-6">
                            <label for="ct_hours">Cycle Time (Hrs)</label>
                            <div class="input-group">
                                <input type="number" class="form-control" id="ct_hours" min="0" required placeholder="0">
                                <div class="input-group-append">
                                    <span class="input-group-text bg-light text-muted">hrs</span>
                                </div>
                            </div>
                        </div>
                        <div class="form-group col-md-6">
                            <label for="ct_minutes">Cycle Time (Mins)</label>
                            <div class="input-group">
                                <input type="number" class="form-control" id="ct_minutes" min="0" max="59" required placeholder="0">
                                <div class="input-group-append">
                                    <span class="input-group-text bg-light text-muted">mins</span>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="row">
                        <div class="form-group col-md-6">
                            <label for="start_date">Expected Start</label>
                            <input type="datetime-local" class="form-control" id="start_date" required>
                        </div>
                        <div class="form-group col-md-6">
                            <label for="end_date">Expected End</label>
                             <input type="datetime-local" class="form-control bg-light" id="end_date" readonly>
                        </div>
                    </div>
                    <hr class="my-3">
                    <div class="row">
                         <div class="form-group col-md-6">
                            <label for="source">Source</label>
                            <select class="form-control" id="source">
                                <option value="inhouse">Inhouse</option>
                                <option value="outsourced">Outsourced</option>
                            </select>
                        </div>
                        <div class="form-group col-md-6">
                            <label for="previous_process">Dependent On (Previous Process)</label>
                            <select class="form-control" id="previous_process" name="previous_process_id">
                                <option value="">None</option>
                                <!-- Populated by JS -->
                            </select>
                        </div>
                    </div>
                </div>
                <div class="modal-footer bg-light" style="border-radius: 0 0 16px 16px;">
                    <button type="button" class="btn btn-secondary font-weight-bold" data-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary font-weight-bold shadow-sm" id="saveProcessBtn">
                        <i class="fas fa-save mr-2"></i>Save Changes
                    </button>
                </div>
            </form>
        </div>
    </div>
@endsection

@push('scripts')
    <script>
window.OPERATOR_MAP = @json($operators);
window.MACHINE_AVAILABILITY = @json($machineAvailability);
window.OPERATOR_AVAILABILITY = @json($operatorAvailability);
let productType = null;
let machineList = @json(\App\Models\Machines::pluck('machine_id')->toArray());
let processes = [];
let currentRC = null;

const RCid = @json($id ?? request()->route('id'));
if (!RCid) window.location.href = "{{ url('superadmin/manufacturing/planning') }}";

// --- PRELOAD DATA ---
const serverRC = @json($routeCard ?? null);
if (serverRC) {
    currentRC = serverRC;
    productType = serverRC.product_type;
    processes = serverRC.processes || [];
    window.RC_PROJECT_START_DATE = serverRC.project_start_date;
}

// --- Utility: Format Dates ---
function formatDateLocal(dateStr) {
    if (!dateStr || dateStr.startsWith('0000')) return '<span class="text-muted">-</span>';
    const d = new Date(dateStr);
    if (isNaN(d.getTime())) return '<span class="text-muted">-</span>';
    return d.toLocaleString('en-GB', {
        day: '2-digit', month: 'short', year: 'numeric',
        hour: '2-digit', minute: '2-digit'
    });
}

function fetchAndSetProcesses(selected, callback) {
     if (!productType) {
        $("#process_name").html('<option value="">No product type found</option>');
        return;
    }
    $.get("{{url('/superadmin/routecard/get-process-list')}}", { product_type: productType }, function(data) {
        let html = '<option value="">Select Process</option>';
        if (data && data.length) {
            data.forEach(p => {
                html += `<option value="${p}" ${selected === p ? "selected" : ""}>${p}</option>`;
            });
            html += '<option value="Other" '+(selected==="Other"?"selected":"")+'>Other</option>';
        } else {
             html += '<option value="Other" '+(selected==="Other"?"selected":"")+'>Other</option>';
        }
        $("#process_name").html(html);
        
        // If the selected value is not in the list (and not "Other"), and it's not empty, append it or handle it.
        // For now, we trust the "selected" logic above or the callback.
        
        if (callback) callback();
    });
}


function populateMachineDropdown(selected) {
    let html = '<option value="">Select Machine</option>';
    machineList.forEach(m => {
        let avail = (window.MACHINE_AVAILABILITY && window.MACHINE_AVAILABILITY[m]) ? (' (' + window.MACHINE_AVAILABILITY[m] + ')') : '';
        html += `<option value="${m}" ${selected === m ? "selected" : ""}>${m}${avail}</option>`;
    });
    $("#machine").html(html);
}

function populateOperatorDropdown(selected) {
    let html = '<option value="">Select Operator</option>';
    Object.entries(window.OPERATOR_MAP).forEach(([oid, oname]) => {
        let avail = (window.OPERATOR_AVAILABILITY && window.OPERATOR_AVAILABILITY[oid]) ? (' (' + window.OPERATOR_AVAILABILITY[oid] + ')') : '';
        html += `<option value="${oid}" ${selected == oid ? "selected" : ""}>${oname}${avail}</option>`;
    });
    $("#operator").html(html);
}

function populatePreviousProcessOptions(currentId) {
    let options = `<option value="">None</option>`;
    processes.forEach(proc => {
        if (!currentId || proc.id != currentId) {
            options += `<option value="${proc.id}">${proc.process || ('Process #' + proc.id)}</option>`;
        }
    });
    $("#previous_process").html(options);
}
let previousMachine = null;
let previousOperator = null;

function setOutsourcedUI(isOutsourced) {
    const $machine = $("#machine");
    const $operator = $("#operator");
    if (isOutsourced) {
        previousMachine = $machine.val();
        previousOperator = $operator.val();
        $machine.prop("disabled", true).val("");
        $operator.prop("disabled", true).val("");
    } else {
        $machine.prop("disabled", false);
        $operator.prop("disabled", false);
        if (previousMachine) $machine.val(previousMachine);
        if (previousOperator) $operator.val(previousOperator);
    }
}

$("#source").on("change", function () {
    setOutsourcedUI($(this).val() === "outsourced");
});


function editProcess(id) {
    let proc = processes.find(p => p.id === id);
    if (!proc) return;
    currentRC = currentRC || {};
    productType = currentRC.product_type || (typeof routeCard !== 'undefined' ? routeCard.product_type : null);

    populatePreviousProcessOptions(proc.id);
    
    // Pass callback to set value AFTER options are loaded
    fetchAndSetProcesses(proc.process, function() {
        $("#process_name").val(proc.process || "");
    });
    
    populateMachineDropdown(proc.machine);
    populateOperatorDropdown(proc.operator);

    $("#operator").val(proc.operator || "");
    $("#source").val(proc.source || "inhouse");
    $("#process_id").val(proc.id);
    $("#description").val(proc.description || '');
    $("#ct_hours").val(proc.ct_hours ?? proc.cycle_hours ?? '');
    $("#ct_minutes").val(proc.ct_minutes ?? proc.cycle_minutes ?? '');
    $("#start_date").val(proc.start_date ? proc.start_date.substring(0, 16) : '');
    $("#end_date").val(proc.end_date ? proc.end_date.substring(0, 16) : '');
    $("#previous_process").val(proc.previous_process_id || '');
    previousMachine = proc.machine || null;
    previousOperator = proc.operator || null;
    setOutsourcedUI((proc.source || "inhouse") === "outsourced");

    // Min Date Logic
     let minDate = '';
    if (proc.previous_process_id) {
        let prev = processes.find(p => String(p.id) === String(proc.previous_process_id));
        if (prev && prev.end_date) {
            const d = new Date(prev.end_date);
            if (!isNaN(d.getTime())) {
                minDate = d.toISOString().substring(0, 16);
            }
        }
    } else if (currentRC && currentRC.project_start_date) {
        const d = new Date(currentRC.project_start_date);
        if (!isNaN(d.getTime())) {
             minDate = d.toISOString().substring(0, 16);
        }
    }
    if (minDate) $("#start_date").attr("min", minDate);
    else $("#start_date").removeAttr("min");

    $("#processModal").modal('show');
}

function fetchProcesses() {
    fetch(`{{ url('superadmin/routecard') }}/${RCid}/processes`)
        .then(res => res.json())
        .then(rc => {
            currentRC = rc;
            productType = rc.product_type;
            processes = rc.processes || [];
            window.RC_PROJECT_START_DATE = rc.project_start_date;
            renderRCHeader(rc);
            renderBoard(processes);
        });
}

function calculateEndDate() {
    let startVal = $("#start_date").val();
    let hours = parseInt($("#ct_hours").val()) || 0;
    let mins = parseInt($("#ct_minutes").val()) || 0;
    if (!startVal) { $("#end_date").val(""); return; }
    let startDate = new Date(startVal);
    if (isNaN(startDate.getTime())) { $("#end_date").val(""); return; }
    startDate.setHours(startDate.getHours() + hours);
    startDate.setMinutes(startDate.getMinutes() + mins);
    
    // Format to YYYY-MM-DDTHH:MM
    let iso = startDate.toLocaleString('sv').replace(' ', 'T').substring(0, 16);
    $("#end_date").val(iso);
}
$("#start_date, #ct_hours, #ct_minutes").on("input change", calculateEndDate);

$("#processForm").submit(function(e) {
    e.preventDefault();
    if ($("#source").val() === "outsourced") {
        $("#machine").val("");
        $("#operator").val("");
    }
    let data = {
        route_card_id: RCid,
        process: $("#process_name").val(),
        description: $("#description").val(),
        machine: $("#machine").val(),
        operator: $("#operator").val(),
        ct_hours: $("#ct_hours").val(),
        ct_minutes: $("#ct_minutes").val(),
        start_date: $("#start_date").val(),
        end_date: $("#end_date").val(),
        source: $("#source").val(),
        previous_process_id: $("#previous_process").val() || null,
    };
    let process_id = $("#process_id").val();
    let method = process_id ? 'PUT' : 'POST';
    let url = process_id ?
        `{{ url('superadmin/routecard/process-reschedule') }}/${process_id}` :
        `{{ url('superadmin/routecard/process') }}`;

    $("#saveProcessBtn").prop('disabled', true).html('<i class="fas fa-spinner fa-spin mr-2"></i>Saving...');

    fetch(url, {
        method: method,
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
        },
        body: JSON.stringify(data)
    })
    .then(res => res.json())
    .then(resp => {
        $("#saveProcessBtn").prop('disabled', false).html('<i class="fas fa-save mr-2"></i>Save Changes');
        if (resp.success) {
            $("#processModal").modal('hide');
            fetchProcesses();
        } else {
            alert(resp.message || "Failed to save process.");
        }
    })
    .catch(() => {
        $("#saveProcessBtn").prop('disabled', false).html('<i class="fas fa-save mr-2"></i>Save Changes');
        alert("Server Error");
    });
});

function statusBadge(status) {
    status = status ? status.toLowerCase() : '';
    let cls = 'secondary';
    let icon = 'circle';
    if (status === 'approved') { cls = 'success'; icon = 'check-circle'; }
    if (status === 'pending') { cls = 'warning'; icon = 'clock'; }
    if (status === 'rejected') { cls = 'danger'; icon = 'times-circle'; }
    if (status === 'in-progress') { cls = 'primary'; icon = 'play-circle'; }
    if (status === 'completed') { cls = 'success'; icon = 'flag-checkered'; }
    
    return `<span class="badge badge-premium badge-${cls}"><i class="fas fa-${icon} mr-1"></i>${status.charAt(0).toUpperCase() + status.slice(1)}</span>`;
}

function renderRCHeader(rc) {
    let html = `
    <div class="card card-premium bg-white mb-4">
        <div class="card-body p-4">
            <div class="row align-items-center mb-3">
                 <div class="col-md-9">
                     <h6 class="text-uppercase text-primary font-weight-bold mb-1 letter-spacing-1">Route Card #${rc.id}</h6>
                     <h2 class="font-weight-bold text-gray-900 mb-0">${rc.company_name || 'Unknown Company'}</h2>
                 </div>
                 <div class="col-md-3 text-md-right mt-3 mt-md-0">
                     ${statusBadge(rc.status)}
                 </div>
            </div>
            <div class="row border-top pt-3 mt-2">
                 <div class="col-6 col-md-3 mb-3 mb-md-0">
                     <div class="info-label">Route No</div>
                     <div class="info-value text-primary">${rc.route_no}</div>
                 </div>
                 <div class="col-6 col-md-3 mb-3 mb-md-0">
                     <div class="info-label">RC Date</div>
                     <div class="info-value"><i class="far fa-calendar-alt mr-1 text-gray-400"></i> ${rc.route_date || '-'}</div>
                 </div>
                 <div class="col-6 col-md-3">
                     <div class="info-label">Approved By</div>
                     <div class="info-value">${rc.approved_by || '<span class="text-muted font-italic">Pending</span>'}</div>
                 </div>
                 <div class="col-6 col-md-3 text-md-right">
                     <div class="info-label">Approval Date</div>
                     <div class="info-value">${formatDateLocal(rc.approved_at)}</div>
                 </div>
            </div>
        </div>
    </div>
    `;
    $("#rcHeader").html(html);
}

function renderBoard(processArr) {
    let board = $(".board");
    board.html('');

    if (!processArr.length) {
        board.html(`
            <div class="card card-premium shadow-sm py-5 px-3 text-center border-0">
                <div class="text-gray-400 mb-3"><i class="fas fa-clipboard-list fa-4x"></i></div>
                <h4 class="text-gray-800 font-weight-bold">No Processes Defined</h4>
                <p class="text-muted mb-0">Use the planning tool to add processes to this route card.</p>
            </div>
        `);
        return;
    }
    
    let html = `<h5 class="mb-3 font-weight-bold text-gray-800 pl-3">Process Timeline</h5>`;
    processArr.forEach(proc => {
        html += processListItem(proc);
    });
    board.append(html);
}

function processListItem(proc) {
    let ct_hours = proc.ct_hours ?? proc.cycle_hours ?? 0;
    let ct_minutes = proc.ct_minutes ?? proc.cycle_minutes ?? 0;
    let ctStr = (ct_hours || ct_minutes) ? `${ct_hours}h ${ct_minutes}m` : '-';

    let operatorName = '-';
    if (proc.operator) {
        operatorName = (window.OPERATOR_MAP && window.OPERATOR_MAP[proc.operator]) ? window.OPERATOR_MAP[proc.operator] : proc.operator;
    }

    let previousName = 'None';
    if (proc.previous_process_id) {
        let prev = processes.find(p => String(p.id) === String(proc.previous_process_id));
        previousName = prev ? (prev.process || ('#' + prev.id)) : 'Unknown';
    }

    // Determine colors based on status/machine
    let isActive = proc.status === 'in-progress';
    let isCompleted = proc.status === 'completed';
    let borderClass = isActive ? 'border-primary' : (isCompleted ? 'border-success' : 'border-left-primary');
    let bgClass = isActive ? 'bg-light-primary' : 'bg-white';

    return `
    <div class="card card-premium process-card shadow-sm ${bgClass}" id="proc-${proc.id}">
        <div class="card-body p-3">
             <div class="d-flex justify-content-between align-items-start">
                 <div>
                     <span class="badge badge-light border mb-2 text-dark">${proc.source === 'outsourced' ? 'Outsourced' : 'In-House'}</span>
                     <h5 class="font-weight-bold text-gray-900 mb-1">${proc.process || 'Untitled Process'}</h5>
                     <p class="text-muted small mb-2">${proc.description || 'No description provided'}</p>
                 </div>
                 <button class="btn btn-light btn-sm shadow-sm text-primary rounded-circle p-2" onclick="editProcess(${proc.id})" title="Edit Process">
                     <i class="fas fa-pen"></i>
                 </button>
             </div>
             <hr class="my-2 dashed" style="border-top: 1px dashed #e3e6f0;">
             <div class="row small mt-2">
                 <div class="col-md-3 mb-2">
                     <div class="text-xs text-uppercase font-weight-bold text-gray-500 mb-1">Assigned To</div>
                     <div class="font-weight-bold text-dark"><i class="fas fa-user-cog mr-1 text-info"></i> ${operatorName}</div>
                     <div class="text-xs text-gray-500 mt-1"><i class="fas fa-cogs mr-1"></i> ${proc.machine || 'No Machine'}</div>
                 </div>
                 <div class="col-md-3 mb-2">
                     <div class="text-xs text-uppercase font-weight-bold text-gray-500 mb-1">Cycle Time</div>
                     <div class="font-weight-bold text-dark"><i class="fas fa-stopwatch mr-1 text-warning"></i> ${ctStr}</div>
                 </div>
                 <div class="col-md-3 mb-2">
                     <div class="text-xs text-uppercase font-weight-bold text-gray-500 mb-1">Schedule</div>
                     <div class="text-nowrap"><span class="text-success">Start:</span> ${formatDateLocal(proc.start_date)}</div>
                     <div class="text-nowrap"><span class="text-danger">End:</span> ${formatDateLocal(proc.end_date)}</div>
                 </div>
                 <div class="col-md-3 mb-2">
                     <div class="text-xs text-uppercase font-weight-bold text-gray-500 mb-1">Dependency</div>
                     <div class="font-weight-bold ${previousName !== 'None' ? 'text-primary' : 'text-muted'}">
                         <i class="fas fa-link mr-1"></i> ${previousName}
                     </div>
                 </div>
             </div>
        </div>
    </div>
    `;
}

$(document).ready(function() {
    if (typeof serverRC !== 'undefined' && serverRC) {
        renderRCHeader(serverRC);
        renderBoard(processes);
    } else {
        fetchProcesses();
    }

    $('#processModal').on('hidden.bs.modal', function() {
        $('#processForm')[0].reset();
        $("#process_name").html('<option value="">Select Process</option>');
        $("#machine").html('<option value="">Select Machine</option>');
        $("#start_date").removeAttr("min");
         $("#processMessage").hide().removeClass('alert alert-success alert-danger').text('');
    });
});
</script>
@endpush
