/**
 * ----------------------------- script for edit button-------------------------------------------------------------------
 */

// Use event delegation for the edit and Convert buttons
document.getElementById('dataTable').addEventListener('click', function(event) {
    if (event.target && event.target.classList.contains('edit-btn')) {
        const rfqID = event.target.closest('button').dataset.id; // Assuming first cell has the rfq ID

        // Store the rfq ID in session storage
        sessionStorage.setItem('currentRFQId', rfqID);

        // Redirect to the form page for editing details
        window.location.href = 'rfq.html';
    }
    if (event.target && event.target.classList.contains('convert-btn')) {
        const rfqQuoteID = event.target.closest('button').dataset.id;
        // Store the rfq ID in session storage
        sessionStorage.setItem('currentRfqQuoteId', rfqQuoteID);
        // Redirect to the form page for editing details
        window.location.href = 'quotation.html';
    }
});


/**
 * ----------------------------- script for "View" button-------------------------------------------------------------------
 */

// Event listener for the "View" button in the RFQ Details table
document.getElementById('dataTable').addEventListener('click', function(event) {
    // Check if the clicked element is the view button
    if (event.target && event.target.id === 'view-btn') {
        const pdfUrl = event.target.dataset.pdfUrl; // Get the PDF URL from the button's data attribute

        if (pdfUrl) {
            // Open the PDF in a new tab
            window.open(pdfUrl, '_blank');
        } else {
            alert("No PDF file found for this entry.");
        }
    }
});

/**
 * ----------------------------- script for import details-------------------------------------------------------------------
 */


// Trigger the modal when the Import button is clicked
document.getElementById('import-button').addEventListener('click', function() {
    $('#importModal').modal('show');
});

// Handle the Excel file import
document.getElementById('importDataBtn').addEventListener('click', async function () {
    let fileInput = document.getElementById('excelFile');
    let file = fileInput.files[0]; // Get the uploaded file

    if (file) {
        let reader = new FileReader();
        
        reader.onload = async function (e) {
            let data = e.target.result;
            let workbook = XLSX.read(data, { type: 'binary' });
            
            // Assuming the first sheet contains the data
            let sheetName = workbook.SheetNames[0];
            let sheet = workbook.Sheets[sheetName];
            
            // Convert sheet to JSON format
            let excelData = XLSX.utils.sheet_to_json(sheet);

            for (const row of excelData) {
                const rfqNo = await generateRFQId();
                if (!rfqNo) continue; // Skip if RFQ No is empty

                // Prepare RFQ data
                const rfqData = {
                    rfqNo: rfqNo,
                    rfqDate: row['RFQ Date'] || '',
                    cusRefNo: row['Customer Reference No'] || '',
                    customer: row['Customer Name'] || '',
                    rfqType: row['RFQ Type'] || '',
                    productType: row['Product Type'] || '',
                    partNo: row['Part No'] || '',
                    projectMaterialNo: row['Project Material No'] || '',
                    drawingNo: row['Drawing No'] || '',
                    drawingRev: row['Drawing Revision'] || '',
                    quantity: row['Quantity'] || '',
                    uom: row['UOM'] || '',
                    note: row['Note'] || '',
                    rawMaterials: [{ name: row['Raw Material'] || '' }],
                };

                // Save RFQ Data to API
                try {
                    const response = await fetch('https://umbreyengineering.com/api/rfq/', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify(rfqData),
                    });

                    if (!response.ok) {
                        throw new Error(`Failed to create RFQ: ${response.statusText}`);
                    }

                    const responseData = await response.json();
                    console.log(`RFQ created successfully:`, responseData);
                } catch (error) {
                    console.error(`Error creating RFQ:`, error);
                }
            }
            loadRFQs();

            // Notify user of successful import
            alert("RFQ details imported successfully!");
        };

        reader.readAsBinaryString(file);
    } else {
        alert("Please choose an Excel file.");
    }
});

// Function to convert Excel numeric date to JavaScript Date
function excelDateToJSDate(serial) {
    const epoch = Date.parse('1899-12-30'); // Excel's epoch start date
    const date = new Date((serial - 25569) * 86400 * 1000); // Convert Excel date serial to JavaScript Date
    return date;
}

/**
 * ----------------------------- script for export-buttons-------------------------------------------------------------------
 */

$(document).ready(function () {

    // Initialize the DataTable
    $('#dataTable').DataTable();

    loadRFQs();

    // Initialize dropdown behavior
    $('#columnVisibility').on('click', function(event) {
        event.preventDefault(); // Prevent default action
        const $dropdownMenu = $(this).next('.dropdown-menu'); // Target the specific dropdown menu
        $('.dropdown-menu').not($dropdownMenu).hide(); // Hide other dropdowns
        $dropdownMenu.toggle(); // Toggle the associated dropdown menu
    });

    // Close dropdown when clicking outside
    $(document).on('click', function(e) {
        if (!$(e.target).closest('.dropdown').length) {
            $('.dropdown-menu').hide(); // Hide all dropdowns
        }
    });

    // Handle column visibility when the checkbox state changes
    $(document).on('change', '.column-toggle', function () {
        updateColumnVisibility();
    });

    // Initial column visibility update
    updateColumnVisibility();

    // Function to copy visible table data
    $("#copy").click(function () {
        const table = document.getElementById("dataTable");
        const rows = table.querySelectorAll("tr");
        let textToCopy = "";

        // Get headers for copying
        const headers = [];
        table.querySelectorAll("thead th").forEach((header) => {
            if ($(header).is(':visible') && header.innerText.trim() !== 'Status') {
                headers.push(header.innerText.trim());
            }
        });
        textToCopy += headers.join("\t") + "\n"; // Add headers

        // Get data
        rows.forEach(row => {
            const cols = row.querySelectorAll("td, th");

            cols.forEach((col) => {
                const headerText = table.querySelectorAll("th")[col.cellIndex].innerText.trim();

                // Exclude the 'Status' column and check for visibility
                if (headerText !== 'Status' && $(col).is(':visible')) {
                    textToCopy += col.innerText.trim() + "\t"; // Add a tab between columns
                }
            });
            textToCopy += "\n"; // New line after each row
        });

        navigator.clipboard.writeText(textToCopy.trim());
        alert("Table data copied to clipboard!");
    });

    // Function to export visible table data as CSV
    $("#export-csv").click(function () {
        let csv = [];
        const rows = document.querySelectorAll("#dataTable tr");

        // Get headers for CSV
        const headers = [];
        document.querySelectorAll("#dataTable thead th").forEach((header) => {
            if ($(header).is(':visible') && header.innerText.trim() !== 'Status') {
                headers.push(header.innerText.trim());
            }
        });
        csv.push(headers.join(",")); // Add header row

        // Get data
        rows.forEach(row => {
            const cols = row.querySelectorAll("td, th");
            const csvRow = [];
            cols.forEach((col) => {
                const headerText = document.querySelectorAll("#dataTable th")[col.cellIndex].innerText.trim();

                // Exclude the 'Status' column and check for visibility
                if (headerText !== 'Status' && $(col).is(':visible')) {
                    csvRow.push(col.innerText.replace(/,/g, " ")); // Replace commas to avoid issues
                }
            });
            if (csvRow.length > 0) { // Add only if the row has data
                csv.push(csvRow.join(",")); // Join the row values with a comma
            }
        });

        const csvString = csv.join("\n"); // Join rows with a new line
        const blob = new Blob([csvString], { type: "text/csv;charset=utf-8;" });
        const link = document.createElement("a");
        link.href = URL.createObjectURL(blob);
        link.setAttribute("download", "rfqdata.csv"); // Set the default filename
        link.click(); // Trigger the download
    });

    // Function to export visible table data as PDF 
    $("#export-pdf").click(function () {
        const { jsPDF } = window.jspdf;
        const doc = new jsPDF({ orientation: 'landscape' }); // Set orientation to landscape
        const table = document.getElementById("dataTable");
        const headers = [];
        const data = [];

        // Get headers from the table
        const headerCols = table.querySelectorAll("thead th");
        headerCols.forEach((header) => {
            const headerText = header.innerText.trim();
            // Exclude the 'Status' column and check for visibility
            if (headerText !== 'Image' && headerText !== 'Status' && $(header).is(':visible')) {
                headers.push(headerText);
            }
        });

        // Get data from the table rows
        const rows = table.querySelectorAll("tbody tr");
        rows.forEach(row => {
            const rowData = [];
            const cols = row.querySelectorAll("td");
            cols.forEach((col) => {
                const headerText = table.querySelectorAll("thead th")[col.cellIndex].innerText.trim();

                // Exclude the 'Status' column by comparing header text and check for visibility
                if (headerText !== 'Image' && headerText !== 'Status' && $(col).is(':visible')) {
                    rowData.push(col.innerText);
                }
            });
            if (rowData.length > 0) { // Add only if the row has data
                data.push(rowData);
            }
        });

        // Generate the PDF using autoTable plugin
        doc.autoTable({
            head: [headers],
            body: data,
            theme: 'grid',
            styles: { fontSize: 10 }
        });

        // Save the PDF
        doc.save('RFQData.pdf');
    });

    // Function to export visible table data as Excel
    $("#export-excel").click(function () {
        const table = document.getElementById("dataTable");
        const wb = XLSX.utils.book_new();
        const ws = XLSX.utils.aoa_to_sheet([]);

        // Get headers from the table, excluding the 'Status' column
        const headerRow = [];
        const headerCells = table.querySelectorAll("thead th");
        headerCells.forEach((headerCell) => {
            const headerText = headerCell.innerText.trim();
            // Exclude the 'Status' column and check for visibility
            if (headerText !== 'Status' && $(headerCell).is(':visible')) {
                headerRow.push(headerText); // Include only relevant headers
            }
        });
        XLSX.utils.sheet_add_aoa(ws, [headerRow], { origin: "A1" });

        // Get data from the table rows, excluding the 'Status' column
        const rows = table.querySelectorAll("tbody tr");
        rows.forEach((row, rowIndex) => {
            const rowData = [];
            const cols = row.querySelectorAll("td");
            cols.forEach((col) => {
                const headerText = table.querySelectorAll("thead th")[col.cellIndex].innerText.trim();

                // Exclude the 'Status' column by comparing header text and check for visibility
                if (headerText !== 'Status' && $(col).is(':visible')) {
                    rowData.push(col.innerText);
                }
            });
            if (rowData.length > 0) { // Add only if the row has data
                XLSX.utils.sheet_add_aoa(ws, [rowData], { origin: `A${rowIndex + 2}` });
            }
        });

        XLSX.utils.book_append_sheet(wb, ws, "Sheet1");
        XLSX.writeFile(wb, "RFQdata.xlsx"); // Set the default filename
    });

    // Function to print the visible table data
    $("#print").click(function () {
        const table = document.getElementById("dataTable");
        const newWindow = window.open("");
        newWindow.document.write("<html><head><title>Print</title></head><body>");
        newWindow.document.write("<table border='1' style='border-collapse: collapse;'>");

        // Get headers for print
        newWindow.document.write("<tr>");
        table.querySelectorAll("thead th").forEach(header => {
            const headerText = header.innerText.trim();
            // Exclude the 'Status' column and check for visibility
            if (headerText !== 'Status' && $(header).is(':visible')) {
                newWindow.document.write(`<th>${headerText}</th>`);
            }
        });
        newWindow.document.write("</tr>");

        // Get data
        const rows = table.querySelectorAll("tbody tr");
        rows.forEach(row => {
            newWindow.document.write("<tr>");
            const cols = row.querySelectorAll("td");
            cols.forEach((col) => {
                const headerText = table.querySelectorAll("thead th")[col.cellIndex].innerText.trim();

                // Exclude the 'Status' column by comparing header text and check for visibility
                if (headerText !== 'Status' && $(col).is(':visible')) {
                    newWindow.document.write(`<td>${col.innerText}</td>`);
                }
            });
            newWindow.document.write("</tr>");
        });

        newWindow.document.write("</table>");
        newWindow.document.write("</body></html>");
        newWindow.document.close();
        newWindow.print();
    });
    
});


// Function to update column visibility based on checkbox state
function updateColumnVisibility() {
    $(".column-toggle").each(function () {
        const columnIndex = $(this).val();
        const isChecked = $(this).is(":checked");
        const table = $("#dataTable"); // Ensure this ID matches your actual table ID

        // Show or hide the column based on checkbox state
        table.find(`tr`).each(function () {
            $(this).find(`td:eq(${columnIndex}), th:eq(${columnIndex})`).css('display', isChecked ? '' : 'none');
        });
    });
}

/**
 * ----------------------------- script for -------------------------------------------------------------------
 */

async function loadRFQs() {
    try {
        // Fetch RFQ data from API
        const response = await fetch('https://umbreyengineering.com/api/rfq/');
        if (!response.ok) {
            throw new Error(`Failed to fetch RFQ data: ${response.statusText}`);
        }
        const rfqDataList = await response.json();

        // Initialize DataTable
        let rfqTable = $('#dataTable').DataTable();
        rfqTable.clear(); // Clear existing table data

        // Populate table with retrieved RFQ data
        rfqDataList.forEach((rfq) => {
            // Format RFQ Date
            const rfqDate = rfq.rfqDate ? new Date(rfq.rfqDate).toLocaleDateString('en-GB') : '';

            // Extract raw materials names
            const rawMaterialsName = rfq.rawMaterials && rfq.rawMaterials.length
                ? rfq.rawMaterials.map((material) => material.name || 'Unknown').join('<br>')
                : '';

            // Add row to DataTable
            rfqTable.row.add([
                rfq.rfqNo || '',
                rfqDate,
                rfq.cusRefNo || '',
                rfq.customer || '',
                rfq.rfqType || '',
                rfq.productType || '',
                rfq.partNo || '',
                rfq.projectMaterialNo || '',
                rfq.drawingNo || '',
                `<button id="view-btn" class="btn btn-success" data-pdf-url="${rfq.files?.drawingFile || ''}">View</button>`,
                rfq.quantity || '',
                rfq.uom || '',
                rfq.note || '',
                rawMaterialsName,
                `<button id="view-btn" class="btn btn-success" data-pdf-url="${rfq.files?.materialSpecificationFile || ''}">View</button>`,
                `<button id="view-btn" class="btn btn-success" data-pdf-url="${rfq.files?.rfqFormFile || ''}">View</button>`,
                `<button id="view-btn" class="btn btn-success" data-pdf-url="${rfq.files?.annexureFile || ''}">View</button>`,
                `<button class="btn btn-danger edit-btn" data-id="${rfq.id}">Edit</button>`,
                `<button class="btn btn-secondary convert-btn" type="button" data-id="${rfq.id}">Quotation</button>`
            ]).draw(false);
        });

    } catch (error) {
        console.error("Error loading RFQ data:", error);
        alert("Failed to load RFQ data.");
    }
}


// Function to generate the RFQ ID using API data
async function generateRFQId() {
    try {
        const response = await fetch('https://umbreyengineering.com/api/rfq/');
        if (!response.ok) {
            throw new Error(`Failed to fetch RFQ data: ${response.statusText}`);
        }
        const rfqDataList = await response.json();

        // Extract the numeric portions of all RFQ IDs in the fetched data
        const allIds = rfqDataList.map(data => {
            const idParts = data.rfqNo?.split('/');
            return idParts && idParts.length === 3 ? parseInt(idParts[2], 10) : 0;
        });

        // Find the largest numeric portion
        const maxIdNumber = Math.max(0, ...allIds);

        // Generate the new ID based on the largest number + 1
        const newIdNumber = (maxIdNumber + 1).toString().padStart(4, '0');

        // Construct and return the new RFQ ID
        return `UEPL/RFQ/${newIdNumber}`;
    } catch (error) {
        console.error("Error generating RFQ ID:", error);
        return null;
    }
}