 document.addEventListener('DOMContentLoaded', function () {
            // Initialize all charts
            initProductionChart();
            initDefectChart();
            initMachineStatusChart();
            initProductionLineChart();
            populateAlerts();
            setupTimeButtons();

            // Simulate real-time data updates
            setInterval(updateDashboardData, 5000);
        });

        function initProductionChart() {
            const ctx = document.getElementById('production-chart').getContext('2d');
            const labels = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul'];
            const data = [6500, 5900, 8000, 8100, 5600, 5500, 12450];

            window.productionChart = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: labels,
                    datasets: [{
                        label: 'Units Produced',
                        data: data,
                        backgroundColor: 'rgba(52, 152, 219, 0.2)',
                        borderColor: 'rgba(52, 152, 219, 1)',
                        borderWidth: 2,
                        tension: 0.4,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: {
                            position: 'top',
                        },
                        tooltip: {
                            mode: 'index',
                            intersect: false,
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            title: {
                                display: true,
                                text: 'Units'
                            }
                        }
                    }
                }
            });
        }

        function initDefectChart() {
            const ctx = document.getElementById('defect-chart').getContext('2d');
            const labels = ['Misalignment', 'Cracks', 'Surface Defects', 'Dimensional Issues', 'Other'];
            const data = [15, 10, 8, 12, 5];
            const backgroundColors = [
                'rgba(231, 76, 60, 0.7)',
                'rgba(241, 196, 15, 0.7)',
                'rgba(52, 152, 219, 0.7)',
                'rgba(155, 89, 182, 0.7)',
                'rgba(149, 165, 166, 0.7)'
            ];

            window.defectChart = new Chart(ctx, {
                type: 'doughnut',
                data: {
                    labels: labels,
                    datasets: [{
                        data: data,
                        backgroundColor: backgroundColors,
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: {
                            position: 'right',
                        },
                        tooltip: {
                            callbacks: {
                                label: function (context) {
                                    const label = context.label || '';
                                    const value = context.raw || 0;
                                    const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                    const percentage = Math.round((value / total) * 100);
                                    return `${label}: ${value} (${percentage}%)`;
                                }
                            }
                        }
                    }
                }
            });
        }

        function initMachineStatusChart() {
            const ctx = document.getElementById('machine-status-chart').getContext('2d');
            const labels = ['Running', 'Idle', 'Maintenance', 'Downtime'];
            const data = [15, 3, 2, 1];
            const backgroundColors = [
                'rgba(46, 204, 113, 0.7)',
                'rgba(241, 196, 15, 0.7)',
                'rgba(52, 152, 219, 0.7)',
                'rgba(231, 76, 60, 0.7)'
            ];

            window.machineStatusChart = new Chart(ctx, {
                type: 'bar',
                data: {
                    labels: labels,
                    datasets: [{
                        label: 'Machines',
                        data: data,
                        backgroundColor: backgroundColors,
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            title: {
                                display: true,
                                text: 'Number of Machines'
                            }
                        }
                    }
                }
            });
        }

        function initProductionLineChart() {
            const ctx = document.getElementById('production-line-chart').getContext('2d');
            const labels = ['Line 1', 'Line 2', 'Line 3', 'Line 4', 'Line 5'];
            const data = [3200, 2800, 4100, 3900, 2450];

            window.productionLineChart = new Chart(ctx, {
                type: 'polarArea',
                data: {
                    labels: labels,
                    datasets: [{
                        label: 'Production by Line',
                        data: data,
                        backgroundColor: [
                            'rgba(52, 152, 219, 0.7)',
                            'rgba(155, 89, 182, 0.7)',
                            'rgba(46, 204, 113, 0.7)',
                            'rgba(241, 196, 15, 0.7)',
                            'rgba(231, 76, 60, 0.7)'
                        ],
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: {
                            position: 'right',
                        }
                    },
                    scales: {
                        r: {
                            pointLabels: {
                                display: true
                            }
                        }
                    }
                }
            });
        }

        function populateAlerts() {
            const alerts = [
                {
                    type: 'danger',
                    title: 'Machine 12 Overheating',
                    time: '10 minutes ago',
                    icon: 'fa-exclamation-triangle'
                },
                {
                    type: 'warning',
                    title: 'Line 3 Speed Reduced',
                    time: '25 minutes ago',
                    icon: 'fa-tachometer-alt'
                },
                {
                    type: 'info',
                    title: 'Preventive Maintenance Due',
                    time: '2 hours ago',
                    icon: 'fa-wrench'
                },
                {
                    type: 'danger',
                    title: 'Quality Alert: Dimensional Variance',
                    time: '3 hours ago',
                    icon: 'fa-ruler-combined'
                },
                {
                    type: 'warning',
                    title: 'Low Raw Material Inventory',
                    time: '5 hours ago',
                    icon: 'fa-box-open'
                }
            ];

            const alertList = document.getElementById('alert-list');
            alertList.innerHTML = '';

            alerts.forEach(alert => {
                const alertItem = document.createElement('div');
                alertItem.className = `alert-item alert-${alert.type}`;
                alertItem.innerHTML = `
            <div class="alert-icon">
                <i class="fas ${alert.icon}"></i>
            </div>
            <div class="alert-info">
                <h4>${alert.title}</h4>
                <p>${alert.time}</p>
            </div>
        `;
                alertList.appendChild(alertItem);
            });
        }

        function setupTimeButtons() {
            const buttons = document.querySelectorAll('.btn-time');
            buttons.forEach(button => {
                button.addEventListener('click', function () {
                    // Remove active class from all buttons in the same group
                    const parent = this.parentElement;
                    parent.querySelectorAll('.btn-time').forEach(btn => {
                        btn.classList.remove('active');
                    });

                    // Add active class to clicked button
                    this.classList.add('active');

                    // Get the period from data attribute
                    const period = this.getAttribute('data-period');

                    // Update charts based on period
                    updateChartsForPeriod(period, parent.closest('.chart-container').querySelector('canvas').id);
                });
            });
        }

        function updateChartsForPeriod(period, chartId) {
            // In a real application, you would fetch new data based on the selected period
            // For this demo, we'll just simulate some changes

            if (chartId === 'production-chart') {
                const labels = {
                    week: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
                    month: ['Week 1', 'Week 2', 'Week 3', 'Week 4'],
                    quarter: ['Month 1', 'Month 2', 'Month 3']
                };

                const data = {
                    week: [1200, 1900, 1700, 2100, 2300, 1900, 1350],
                    month: [8200, 9500, 7800, 12450],
                    quarter: [24500, 28700, 32450]
                };

                window.productionChart.data.labels = labels[period] || labels.week;
                window.productionChart.data.datasets[0].data = data[period] || data.week;
                window.productionChart.update();
            } else if (chartId === 'defect-chart') {
                // For defect chart, we might change the data based on period
                const data = {
                    week: [8, 5, 3, 7, 2],
                    month: [15, 10, 8, 12, 5]
                };

                window.defectChart.data.datasets[0].data = data[period] || data.week;
                window.defectChart.update();
            }
        }

        function updateDashboardData() {
            // Simulate real-time data updates
            const randomFactor = Math.random() * 0.2 - 0.1; // -10% to +10%

            // Update card data
            const totalProduction = document.getElementById('total-production');
            const currentProduction = parseInt(totalProduction.textContent.replace(/,/g, ''));
            const newProduction = Math.max(0, currentProduction + Math.floor(currentProduction * randomFactor));
            totalProduction.textContent = newProduction.toLocaleString();

            const machineUtilization = document.getElementById('machine-utilization');
            const currentUtilization = parseInt(machineUtilization.textContent);
            const newUtilization = Math.min(100, Math.max(0, currentUtilization + Math.floor(currentUtilization * randomFactor)));
            machineUtilization.textContent = `${newUtilization}%`;

            const qualityRate = document.getElementById('quality-rate');
            const currentQuality = parseFloat(qualityRate.textContent);
            const newQuality = Math.min(100, Math.max(80, currentQuality + (currentQuality * randomFactor)));
            qualityRate.textContent = `${newQuality.toFixed(1)}%`;

            const oeeRate = document.getElementById('oee-rate');
            const currentOEE = parseInt(oeeRate.textContent);
            const newOEE = Math.min(100, Math.max(50, currentOEE + Math.floor(currentOEE * randomFactor)));
            oeeRate.textContent = `${newOEE}%`;

            // Update production chart with new data point
            if (window.productionChart) {
                const lastValue = window.productionChart.data.datasets[0].data.slice(-1)[0];
                const newValue = Math.max(0, lastValue + Math.floor(lastValue * randomFactor));

                // Shift all data left and add new value
                const labels = window.productionChart.data.labels;
                const data = window.productionChart.data.datasets[0].data;

                if (labels.length > 7) {
                    labels.shift();
                    data.shift();
                }

                // Add new label (time-based in real app)
                const lastLabel = labels[labels.length - 1];
                let newLabel;

                if (typeof lastLabel === 'number') {
                    newLabel = lastLabel + 1;
                } else if (lastLabel.match(/^Week \d+$/)) {
                    const weekNum = parseInt(lastLabel.split(' ')[1]);
                    newLabel = `Week ${weekNum + 1}`;
                } else if (lastLabel.match(/^Month \d+$/)) {
                    const monthNum = parseInt(lastLabel.split(' ')[1]);
                    newLabel = `Month ${monthNum + 1}`;
                } else {
                    // For days
                    const days = ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'];
                    const currentIndex = days.indexOf(lastLabel);
                    newLabel = currentIndex < days.length - 1 ? days[currentIndex + 1] : days[0];
                }

                labels.push(newLabel);
                data.push(newValue);

                window.productionChart.update();
            }

            // Randomly add a new alert occasionally
            if (Math.random() > 0.8) {
                const alertTypes = [
                    { type: 'danger', icon: 'fa-exclamation-triangle', title: 'Machine Overload' },
                    { type: 'warning', icon: 'fa-bolt', title: 'Power Fluctuation' },
                    { type: 'info', icon: 'fa-info-circle', title: 'Scheduled Maintenance' }
                ];

                const randomAlert = alertTypes[Math.floor(Math.random() * alertTypes.length)];
                const alertList = document.getElementById('alert-list');

                const alertItem = document.createElement('div');
                alertItem.className = `alert-item alert-${randomAlert.type}`;
                alertItem.innerHTML = `
            <div class="alert-icon">
                <i class="fas ${randomAlert.icon}"></i>
            </div>
            <div class="alert-info">
                <h4>${randomAlert.title}</h4>
                <p>Just now</p>
            </div>
        `;

                // Insert at top
                if (alertList.firstChild) {
                    alertList.insertBefore(alertItem, alertList.firstChild);
                } else {
                    alertList.appendChild(alertItem);
                }

                // Limit to 10 alerts
                if (alertList.children.length > 10) {
                    alertList.removeChild(alertList.lastChild);
                }
            }
        }