<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class UserActivity extends Model
{
    protected $fillable = [
        'user_id',
        'activity_type',
        'page_title',
        'url',
        'route_name',
        'method',
        'ip_address',
        'user_agent',
        'browser',
        'platform',
        'device',
        'request_data',
        'response_status',
        'response_time',
    ];

    protected $casts = [
        'request_data' => 'array',
        'response_time' => 'decimal:2',
    ];

    /**
     * Activity type constants
     */
    const TYPE_PAGE_VISIT = 'page_visit';
    const TYPE_LOGIN = 'login';
    const TYPE_LOGOUT = 'logout';
    const TYPE_CREATE = 'create';
    const TYPE_UPDATE = 'update';
    const TYPE_DELETE = 'delete';

    /**
     * Get the user that performed the activity.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Scope to filter by user
     */
    public function scopeByUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    /**
     * Scope to filter by activity type
     */
    public function scopeByType($query, $type)
    {
        return $query->where('activity_type', $type);
    }

    /**
     * Scope to filter by date range
     */
    public function scopeBetweenDates($query, $startDate, $endDate)
    {
        return $query->whereBetween('created_at', [$startDate, $endDate]);
    }

    /**
     * Scope to get activities from last N days
     */
    public function scopeLastDays($query, $days = 30)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }

    /**
     * Scope to filter by method
     */
    public function scopeByMethod($query, $method)
    {
        return $query->where('method', strtoupper($method));
    }

    /**
     * Get activity type badge class
     */
    public function getTypeBadgeClassAttribute(): string
    {
        return match($this->activity_type) {
            self::TYPE_LOGIN => 'badge-success',
            self::TYPE_LOGOUT => 'badge-secondary',
            self::TYPE_CREATE => 'badge-primary',
            self::TYPE_UPDATE => 'badge-warning',
            self::TYPE_DELETE => 'badge-danger',
            default => 'badge-info',
        };
    }

    /**
     * Get activity type icon
     */
    public function getTypeIconAttribute(): string
    {
        return match($this->activity_type) {
            self::TYPE_LOGIN => 'fa-sign-in-alt',
            self::TYPE_LOGOUT => 'fa-sign-out-alt',
            self::TYPE_CREATE => 'fa-plus-circle',
            self::TYPE_UPDATE => 'fa-edit',
            self::TYPE_DELETE => 'fa-trash-alt',
            default => 'fa-eye',
        };
    }

    /**
     * Get method badge class
     */
    public function getMethodBadgeClassAttribute(): string
    {
        return match($this->method) {
            'GET' => 'badge-info',
            'POST' => 'badge-success',
            'PUT', 'PATCH' => 'badge-warning',
            'DELETE' => 'badge-danger',
            default => 'badge-secondary',
        };
    }

    /**
     * Log a new activity
     */
    public static function log(array $data): self
    {
        return self::create($data);
    }

    /**
     * Delete activities older than specified days
     */
    public static function deleteOlderThan(int $days = 30): int
    {
        return self::where('created_at', '<', now()->subDays($days))->delete();
    }
}
