@extends('layouts.dashboard.superadminapp')

@section('title', 'Create Sales Order')

@section('styles')
<style>
    .section-card {
        border: none;
        border-radius: 0.5rem;
        box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.1);
        margin-bottom: 1.5rem;
    }
    .section-card .card-header {
        background: linear-gradient(135deg, #4e73df 0%, #224abe 100%);
        color: #fff;
        font-weight: 600;
        padding: 0.75rem 1rem;
        border-radius: 0.5rem 0.5rem 0 0 !important;
        font-size: 0.9rem;
    }
    .section-card .card-header.bg-success-gradient {
        background: linear-gradient(135deg, #1cc88a 0%, #13855c 100%);
    }
    .section-card .card-header.bg-info-gradient {
        background: linear-gradient(135deg, #36b9cc 0%, #258391 100%);
    }
    .section-card .card-header.bg-warning-gradient {
        background: linear-gradient(135deg, #f6c23e 0%, #dda20a 100%);
    }
    .section-card .card-header.bg-secondary-gradient {
        background: linear-gradient(135deg, #858796 0%, #60616f 100%);
    }
    .form-group label {
        font-weight: 600;
        font-size: 0.85rem;
        color: #5a5c69;
        margin-bottom: 0.25rem;
    }
    .form-control, .form-control-sm {
        font-size: 0.85rem;
    }
    .required {
        color: #e74a3b;
    }
    .product-table {
        width: 100%;
        font-size: 0.8rem;
    }
    .product-table th {
        background-color: #f8f9fc;
        font-weight: 600;
        font-size: 0.75rem;
        text-align: center;
        padding: 0.5rem;
        white-space: nowrap;
    }
    .product-table td {
        padding: 0.35rem;
        vertical-align: middle;
    }
    .product-table input, .product-table textarea {
        font-size: 0.8rem;
        padding: 0.25rem 0.5rem;
    }
    .product-table .btn-sm {
        padding: 0.2rem 0.4rem;
        font-size: 0.7rem;
    }
    .totals-section {
        background: #f8f9fc;
        border-radius: 0.5rem;
        padding: 1rem;
    }
    .totals-section .form-group {
        margin-bottom: 0.5rem;
    }
    .totals-section label {
        font-size: 0.8rem;
    }
    .totals-section input {
        text-align: right;
        font-weight: 600;
    }
</style>
@endsection

@section('content')
    <!-- Page Heading -->
    <div class="bg-white p-2 d-sm-flex align-items-center justify-content-between mb-4">
        <h1 class="h5 mb-0 text-gray-800">UEPL | Create Sales Order</h1>
        <div class="display-date">
            <span id="day">day</span>,
            <span id="daynum">00</span>
            <span id="month">month</span>
            <span id="year">0000</span>
        </div>
        <div class="display-time"></div>
    </div>

    <!-- Alerts -->
    @if (session('success'))
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fas fa-check-circle mr-2"></i>{{ session('success') }}
            <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                <span aria-hidden="true">&times;</span>
            </button>
        </div>
    @endif
    @if (session('error'))
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-circle mr-2"></i>{{ session('error') }}
            <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                <span aria-hidden="true">&times;</span>
            </button>
        </div>
    @endif

    <form id="soForm">
        <!-- Sales Order & Quotation Information -->
        <div class="card section-card">
            <div class="card-header">
                <i class="fas fa-file-invoice mr-2"></i> Sales Order & Quotation Information
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="sOrderNo">Sales Order No <span class="required">*</span></label>
                            <input type="text" class="form-control" id="sOrderNo" readonly required>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="sOrderDate">Sales Order Date <span class="required">*</span></label>
                            <input type="date" class="form-control" id="sOrderDate" required>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="quotationSelect">Quotation</label>
                            <select class="form-control selectpicker" id="quotationSelect" data-live-search="true">
                                <option value="">Select Quotation</option>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="quotationDate">Quotation Date</label>
                            <input type="date" class="form-control" id="quotationDate">
                        </div>
                    </div>
                </div>
                <div class="row">
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="grnNo">GRN No</label>
                            <input type="text" class="form-control" id="grnNo" placeholder="GRN Number">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="custRefNo">Customer Reference No</label>
                            <input type="text" class="form-control" id="custRefNo" placeholder="Customer Ref No">
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="customerSO">Customer SO Document</label>
                            <input type="file" class="form-control" id="customerSO" name="customerSO" accept=".pdf">
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Customer Information -->
        <div class="card section-card">
            <div class="card-header bg-success-gradient">
                <i class="fas fa-user mr-2"></i> Customer Information
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="soCompanyId">Company Name</label>
                            <input type="text" class="form-control" id="soCompanyId" placeholder="Company Name" readonly>
                            <input type="hidden" id="soCompanyName">
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="soCompanyAddress">Address</label>
                            <input type="text" class="form-control" id="soCompanyAddress" placeholder="Company Address">
                        </div>
                    </div>
                </div>
                <div class="row">
                    <div class="col-md-4">
                        <div class="form-group">
                            <label for="soCompanyGSTN">GST No</label>
                            <input type="text" class="form-control" id="soCompanyGSTN" placeholder="GSTIN">
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="form-group">
                            <label for="soCompanyPhone">Phone</label>
                            <input type="tel" class="form-control" id="soCompanyPhone" placeholder="Contact Number">
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="form-group">
                            <label for="soCompanyEmail">Email</label>
                            <input type="email" class="form-control" id="soCompanyEmail" placeholder="Email Address">
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- RFQ & Product Information -->
        <div class="card section-card">
            <div class="card-header bg-info-gradient">
                <i class="fas fa-box mr-2"></i> RFQ & Product Information
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="rfqNo">RFQ No</label>
                            <input type="text" class="form-control" id="rfqNo" placeholder="RFQ Number">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="rfqDate">RFQ Date</label>
                            <input type="date" class="form-control" id="rfqDate">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="partNo">Part No</label>
                            <input type="text" class="form-control" id="partNo" placeholder="Part Number">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="projectMaterialNo">Project/Material No</label>
                            <input type="text" class="form-control" id="projectMaterialNo" placeholder="Project Material No">
                        </div>
                    </div>
                </div>
                <div class="row">
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="drawingId">Drawing No</label>
                            <input type="text" class="form-control" id="drawingId" placeholder="Drawing Number">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="drawingRevision">Drawing Revision</label>
                            <input type="text" class="form-control" id="drawingRevision" placeholder="Revision">
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="productType">Product Type</label>
                            <input type="text" class="form-control" id="productType" placeholder="Product Type">
                        </div>
                    </div>
                </div>
                <div class="row">
                    <div class="col-md-12">
                        <div class="form-group">
                            <label for="projectDescription">Description / Notes</label>
                            <textarea class="form-control" id="projectDescription" rows="2" placeholder="Project Description"></textarea>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Billing & Shipping -->
        <div class="card section-card">
            <div class="card-header bg-secondary-gradient">
                <i class="fas fa-shipping-fast mr-2"></i> Billing & Shipping Information
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="billAddress">Bill To</label>
                            <textarea class="form-control" id="billAddress" rows="3" placeholder="Billing Address"></textarea>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="shipAddress">Ship To</label>
                            <textarea class="form-control" id="shipAddress" rows="3" placeholder="Shipping Address"></textarea>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Product Items Table -->
        <div class="card section-card">
            <div class="card-header bg-warning-gradient">
                <i class="fas fa-list mr-2"></i> Product Items
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table id="productTable" class="table table-bordered product-table">
                        <thead>
                            <tr>
                                <th style="width:3%">#</th>
                                <th style="width:8%">Material Code</th>
                                <th style="width:6%">HSN Code</th>
                                <th style="width:12%">Description</th>
                                <th style="width:10%">Additional Details</th>
                                <th style="width:8%">Req By Date</th>
                                <th style="width:5%">Qty</th>
                                <th style="width:5%">UOM</th>
                                <th style="width:7%">Unit Rate</th>
                                <th style="width:5%">Disc %</th>
                                <th style="width:7%">Value</th>
                                <th style="width:5%">SGST %</th>
                                <th style="width:5%">CGST %</th>
                                <th style="width:5%">IGST %</th>
                                <th style="width:7%">Amount</th>
                                <th style="width:4%"></th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr>
                                <td class="text-center">1</td>
                                <td><input type="text" class="form-control form-control-sm material-code"></td>
                                <td><input type="text" class="form-control form-control-sm hsn-code"></td>
                                <td><textarea class="form-control form-control-sm description" rows="1"></textarea></td>
                                <td><textarea class="form-control form-control-sm additional-details" rows="1"></textarea></td>
                                <td><input type="date" class="form-control form-control-sm req-by-date"></td>
                                <td><input type="number" class="form-control form-control-sm quantity" min="0"></td>
                                <td><input type="text" class="form-control form-control-sm uom"></td>
                                <td><input type="number" class="form-control form-control-sm rate" step="0.01" min="0"></td>
                                <td><input type="number" class="form-control form-control-sm discount" step="0.01" min="0" max="100"></td>
                                <td><input type="number" class="form-control form-control-sm value" readonly></td>
                                <td><input type="number" class="form-control form-control-sm sgst" step="0.01" min="0"></td>
                                <td><input type="number" class="form-control form-control-sm cgst" step="0.01" min="0"></td>
                                <td><input type="number" class="form-control form-control-sm igst" step="0.01" min="0"></td>
                                <td><input type="number" class="form-control form-control-sm amount" readonly></td>
                                <td class="text-center">
                                    <button type="button" class="btn btn-danger btn-sm" onclick="deleteRow(this)">
                                        <i class="fa fa-trash"></i>
                                    </button>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
                <button type="button" class="btn btn-outline-primary btn-sm mt-2" id="addRowBtn">
                    <i class="fas fa-plus mr-1"></i> Add Row
                </button>
            </div>
        </div>

        <!-- Totals Section -->
        <div class="card section-card">
            <div class="card-header">
                <i class="fas fa-rupee-sign mr-2"></i> Summary & Totals
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="GT-in-letters">Grand Total in Words (INR)</label>
                            <input type="text" class="form-control" id="GT-in-letters" readonly>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="totals-section">
                            <div class="form-group row align-items-center">
                                <label class="col-md-4 col-form-label">Sub Total</label>
                                <div class="col-md-8">
                                    <input type="text" class="form-control" id="totalAmount" readonly>
                                </div>
                            </div>
                            <div class="form-group row align-items-center">
                                <label class="col-md-4 col-form-label">SGST Amount</label>
                                <div class="col-md-8">
                                    <input type="text" class="form-control" id="sgstA" readonly>
                                </div>
                            </div>
                            <div class="form-group row align-items-center">
                                <label class="col-md-4 col-form-label">CGST Amount</label>
                                <div class="col-md-8">
                                    <input type="text" class="form-control" id="cgstA" readonly>
                                </div>
                            </div>
                            <div class="form-group row align-items-center">
                                <label class="col-md-4 col-form-label">IGST Amount</label>
                                <div class="col-md-8">
                                    <input type="text" class="form-control" id="IgstA" readonly>
                                </div>
                            </div>
                            <div class="form-group row align-items-center">
                                <label class="col-md-4 col-form-label">Round Up</label>
                                <div class="col-md-8">
                                    <input type="text" class="form-control" id="roundUp" readonly>
                                </div>
                            </div>
                            <hr>
                            <div class="form-group row align-items-center mb-0">
                                <label class="col-md-4 col-form-label font-weight-bold text-primary">Grand Total</label>
                                <div class="col-md-8">
                                    <input type="text" class="form-control form-control-lg font-weight-bold text-primary" id="grandTotal" readonly>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Action Buttons -->
        <div class="card section-card">
            <div class="card-body">
                <div class="d-flex justify-content-between">
                    <a href="{{ route('superadmin.sales.salesdetails') }}" class="btn btn-secondary">
                        <i class="fas fa-arrow-left mr-1"></i> Back to List
                    </a>
                    <div>
                        <button type="button" class="btn btn-danger mr-2" onclick="resetForm()">
                            <i class="fas fa-times mr-1"></i> Clear
                        </button>
                        <button type="submit" class="btn btn-success" id="submitBtn">
                            <span class="spinner-border spinner-border-sm d-none" role="status"></span>
                            <span class="btn-text"><i class="fas fa-save mr-1"></i> Save Sales Order</span>
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </form>
@endsection

@section('scripts')
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script>
$(document).ready(function() {
    // Initialize selectpicker
    $('.selectpicker').selectpicker();

    // Set today's date
    document.getElementById('sOrderDate').value = new Date().toISOString().split('T')[0];

    // Generate SO Number
    generateSONumber();

    // Load Quotations
    loadQuotations();

    // Setup initial row event listeners
    setupRowEventListeners(document.querySelector("#productTable tbody tr"));

    // Add Row button
    $('#addRowBtn').click(function() {
        addRow();
    });

    // Form submit
    $('#soForm').on('submit', function(e) {
        e.preventDefault();
        saveSO();
    });

    // Quotation select change
    $('#quotationSelect').on('change', function() {
        loadQuotationDetails(this.value);
    });

    // Calculate totals initially
    calculateAllTotals();
});

// Generate SO Number
async function generateSONumber() {
    try {
        const response = await fetch("{{ url('superadmin/generate-so-id') }}");
        const data = await response.json();
        document.getElementById('sOrderNo').value = data.so_id;
    } catch (error) {
        document.getElementById('sOrderNo').value = 'UEPL/SO/0001';
    }
}

// Load Quotations
async function loadQuotations() {
    try {
        const response = await fetch("{{ url('/superadmin/quotations') }}");
        const quotations = await response.json();
        const select = document.getElementById("quotationSelect");
        quotations.forEach(q => {
            const opt = document.createElement("option");
            opt.value = q.id;
            opt.textContent = q.quotation_no;
            select.appendChild(opt);
        });
        $('.selectpicker').selectpicker('refresh');
    } catch (err) {
        console.error("Failed to load quotations:", err);
    }
}

// Load Quotation Details
async function loadQuotationDetails(quotationId) {
    if (!quotationId) return;
    try {
        const response = await fetch(`{{ url('/superadmin/quotation') }}/${quotationId}`);
        const data = await response.json();
        if (data.error) {
            Swal.fire('Error', data.error, 'error');
            return;
        }

        // Fill form fields
        document.getElementById("quotationDate").value = data.quotation_date || "";
        document.getElementById("soCompanyId").value = data.customer?.company || "";
        document.getElementById("soCompanyName").value = data.customer?.id || "";
        document.getElementById("soCompanyAddress").value = data.address || "";
        document.getElementById("soCompanyGSTN").value = data.gst || "";
        document.getElementById("soCompanyPhone").value = data.phone || "";
        document.getElementById("soCompanyEmail").value = data.email || "";
        document.getElementById("billAddress").value = data.address || "";
        document.getElementById("shipAddress").value = data.address || "";
        document.getElementById("custRefNo").value = data.cust_ref_no || "";
        document.getElementById("rfqNo").value = data.rfq?.rfq_no || "";
        document.getElementById("rfqDate").value = data.rfq_date || "";
        document.getElementById("partNo").value = data.part_no || "";
        document.getElementById("projectMaterialNo").value = data.project_material_no || "";
        document.getElementById("drawingId").value = data.drawing_no || "";
        document.getElementById("drawingRevision").value = data.drawing_rev || "";
        document.getElementById("projectDescription").value = data.note || "";
        document.getElementById("productType").value = data.product_type || "";

        // Fill first row with description and quantity
        const firstRow = document.querySelector("#productTable tbody tr");
        if (firstRow) {
            firstRow.querySelector(".description").value = data.description || "";
            firstRow.querySelector(".quantity").value = data.quantity || "";
            firstRow.querySelector(".uom").value = data.quantity_uom || "";
        }

        calculateAllTotals();
    } catch (err) {
        console.error("Failed to load quotation data:", err);
    }
}

// Add Row
function addRow() {
    const tbody = document.querySelector("#productTable tbody");
    const rowCount = tbody.querySelectorAll('tr').length;
    const newRow = document.createElement("tr");
    newRow.innerHTML = `
        <td class="text-center">${rowCount + 1}</td>
        <td><input type="text" class="form-control form-control-sm material-code"></td>
        <td><input type="text" class="form-control form-control-sm hsn-code"></td>
        <td><textarea class="form-control form-control-sm description" rows="1"></textarea></td>
        <td><textarea class="form-control form-control-sm additional-details" rows="1"></textarea></td>
        <td><input type="date" class="form-control form-control-sm req-by-date"></td>
        <td><input type="number" class="form-control form-control-sm quantity" min="0"></td>
        <td><input type="text" class="form-control form-control-sm uom"></td>
        <td><input type="number" class="form-control form-control-sm rate" step="0.01" min="0"></td>
        <td><input type="number" class="form-control form-control-sm discount" step="0.01" min="0" max="100"></td>
        <td><input type="number" class="form-control form-control-sm value" readonly></td>
        <td><input type="number" class="form-control form-control-sm sgst" step="0.01" min="0"></td>
        <td><input type="number" class="form-control form-control-sm cgst" step="0.01" min="0"></td>
        <td><input type="number" class="form-control form-control-sm igst" step="0.01" min="0"></td>
        <td><input type="number" class="form-control form-control-sm amount" readonly></td>
        <td class="text-center">
            <button type="button" class="btn btn-danger btn-sm" onclick="deleteRow(this)">
                <i class="fa fa-trash"></i>
            </button>
        </td>
    `;
    tbody.appendChild(newRow);
    setupRowEventListeners(newRow);
}

// Delete Row
function deleteRow(button) {
    const row = button.closest("tr");
    row.parentNode.removeChild(row);
    updateRowNumbers();
    calculateAllTotals();
}

// Update Row Numbers
function updateRowNumbers() {
    const rows = document.querySelectorAll("#productTable tbody tr");
    rows.forEach((row, i) => row.cells[0].textContent = i + 1);
}

// Setup Row Event Listeners
function setupRowEventListeners(row) {
    const inputs = ['quantity', 'rate', 'discount', 'sgst', 'cgst', 'igst'];
    inputs.forEach(className => {
        const input = row.querySelector(`.${className}`);
        if (input) {
            input.addEventListener('input', function() {
                calculateRowValue(row);
            });
        }
    });
}

// Calculate Row Value
function calculateRowValue(row) {
    const quantity = parseFloat(row.querySelector('.quantity').value) || 0;
    const rate = parseFloat(row.querySelector('.rate').value) || 0;
    const discount = parseFloat(row.querySelector('.discount').value) || 0;
    const sgst = parseFloat(row.querySelector('.sgst').value) || 0;
    const cgst = parseFloat(row.querySelector('.cgst').value) || 0;
    const igst = parseFloat(row.querySelector('.igst').value) || 0;

    const baseValue = quantity * rate * (1 - discount / 100);
    row.querySelector('.value').value = baseValue.toFixed(2);

    const amount = baseValue * (1 + (sgst + cgst + igst) / 100);
    row.querySelector('.amount').value = amount.toFixed(2);

    calculateAllTotals();
}

// Calculate All Totals
function calculateAllTotals() {
    let subtotal = 0, sgstTotal = 0, cgstTotal = 0, igstTotal = 0;

    document.querySelectorAll("#productTable tbody tr").forEach(row => {
        const val = parseFloat(row.querySelector('.value').value) || 0;
        const sgst = parseFloat(row.querySelector('.sgst').value) || 0;
        const cgst = parseFloat(row.querySelector('.cgst').value) || 0;
        const igst = parseFloat(row.querySelector('.igst').value) || 0;

        subtotal += val;
        sgstTotal += val * sgst / 100;
        cgstTotal += val * cgst / 100;
        igstTotal += val * igst / 100;
    });

    const grandTotalRaw = subtotal + sgstTotal + cgstTotal + igstTotal;
    const roundUp = Math.ceil(grandTotalRaw) - grandTotalRaw;
    const grandTotal = grandTotalRaw + roundUp;

    document.getElementById("totalAmount").value = subtotal.toFixed(2);
    document.getElementById("sgstA").value = sgstTotal.toFixed(2);
    document.getElementById("cgstA").value = cgstTotal.toFixed(2);
    document.getElementById("IgstA").value = igstTotal.toFixed(2);
    document.getElementById("roundUp").value = roundUp.toFixed(2);
    document.getElementById("grandTotal").value = grandTotal.toFixed(2);
    document.getElementById("GT-in-letters").value = numberToWords(Math.round(grandTotal)) + ' Rupees Only';
}

// Get Table Data
function getTableData() {
    const rows = document.querySelectorAll("#productTable tbody tr");
    const data = [];
    rows.forEach(row => {
        data.push({
            slNo: row.cells[0].textContent,
            materialCode: row.querySelector('.material-code').value,
            hsnCode: row.querySelector('.hsn-code').value,
            description: row.querySelector('.description').value,
            additionalDetails: row.querySelector('.additional-details').value,
            reqByDate: row.querySelector('.req-by-date').value,
            quantity: row.querySelector('.quantity').value,
            uom: row.querySelector('.uom').value,
            unitRate: row.querySelector('.rate').value,
            discount: row.querySelector('.discount').value,
            value: row.querySelector('.value').value,
            sgst: row.querySelector('.sgst').value,
            cgst: row.querySelector('.cgst').value,
            igst: row.querySelector('.igst').value,
            amount: row.querySelector('.amount').value
        });
    });
    return data;
}

// Save SO
async function saveSO() {
    const $btn = $('#submitBtn');
    const $spinner = $btn.find('.spinner-border');
    const $text = $btn.find('.btn-text');

    // Validation
    if (!document.getElementById('sOrderNo').value || !document.getElementById('sOrderDate').value) {
        Swal.fire('Error', 'Sales Order No and Date are required!', 'error');
        return;
    }

    $btn.prop('disabled', true);
    $spinner.removeClass('d-none');
    $text.html('Saving...');

    const formData = {
        soNo1: document.getElementById('sOrderNo').value,
        soDate: document.getElementById('sOrderDate').value,
        grnNo1: document.getElementById('grnNo').value,
        custRefSoNo: document.getElementById('custRefNo').value,
        companyName: document.getElementById('soCompanyName').value,
        companyId: document.getElementById('soCompanyId').value,
        soCompanyAddress: document.getElementById('soCompanyAddress').value,
        soCompanyGSTN: document.getElementById('soCompanyGSTN').value,
        soCompanyPhone: document.getElementById('soCompanyPhone').value,
        soCompanyEmail: document.getElementById('soCompanyEmail').value,
        quotationNo2: document.getElementById('quotationSelect').selectedOptions[0]?.text !== 'Select Quotation'
            ? document.getElementById('quotationSelect').selectedOptions[0]?.text
            : null,
        quotationId: document.getElementById('quotationSelect').value,
        quotationDate: document.getElementById('quotationDate').value,
        billAddress: document.getElementById('billAddress').value,
        shipAddress: document.getElementById('shipAddress').value,
        productType: document.getElementById('productType').value,
        totalAmount: document.getElementById('totalAmount').value,
        sgstA: document.getElementById('sgstA').value,
        cgstA: document.getElementById('cgstA').value,
        IgstA: document.getElementById('IgstA').value,
        roundUp: document.getElementById('roundUp').value,
        grandTotal: document.getElementById('grandTotal').value,
        GT_in_letters: document.getElementById('GT-in-letters').value,
        rfqNo: document.getElementById('rfqNo').value,
        rfqDate: document.getElementById('rfqDate').value,
        partNo: document.getElementById('partNo').value,
        projectMaterialNo: document.getElementById('projectMaterialNo').value,
        drawingId: document.getElementById('drawingId').value,
        drawingRevision: document.getElementById('drawingRevision').value,
        projectDescription: document.getElementById('projectDescription').value,
        tableProducts: getTableData()
    };

    try {
        const response = await fetch("{{ route('superadmin.sales.order.store') }}", {
            method: "POST",
            headers: {
                "Content-Type": "application/json",
                "X-CSRF-TOKEN": document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify(formData)
        });

        const data = await response.json();

        if (data.message && data.message.includes('Sales Order saved')) {
            Swal.fire('Success!', data.message, 'success').then(() => {
                window.location.href = "{{ route('superadmin.sales.salesdetails') }}";
            });
        } else {
            Swal.fire('Error!', data.message || data.error || 'Something went wrong.', 'error');
        }
    } catch (err) {
        Swal.fire('Error!', 'Failed to save sales order.', 'error');
        console.error("Save error:", err);
    } finally {
        $btn.prop('disabled', false);
        $spinner.addClass('d-none');
        $text.html('<i class="fas fa-save mr-1"></i> Save Sales Order');
    }
}

// Reset Form
function resetForm() {
    Swal.fire({
        title: 'Clear Form?',
        text: 'Are you sure you want to clear all fields?',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#e74a3b',
        cancelButtonColor: '#858796',
        confirmButtonText: 'Yes, clear it!'
    }).then((result) => {
        if (result.isConfirmed) {
            document.getElementById('soForm').reset();
            generateSONumber();
            document.getElementById('sOrderDate').value = new Date().toISOString().split('T')[0];
            $('.selectpicker').selectpicker('refresh');
            calculateAllTotals();
        }
    });
}

// Number to Words (Indian Format)
function numberToWords(num) {
    if (num === 0) return 'Zero';

    const ones = ['', 'One', 'Two', 'Three', 'Four', 'Five', 'Six', 'Seven', 'Eight', 'Nine',
                  'Ten', 'Eleven', 'Twelve', 'Thirteen', 'Fourteen', 'Fifteen', 'Sixteen',
                  'Seventeen', 'Eighteen', 'Nineteen'];
    const tens = ['', '', 'Twenty', 'Thirty', 'Forty', 'Fifty', 'Sixty', 'Seventy', 'Eighty', 'Ninety'];

    function convertLessThanHundred(n) {
        if (n < 20) return ones[n];
        return tens[Math.floor(n / 10)] + (n % 10 ? ' ' + ones[n % 10] : '');
    }

    function convertLessThanThousand(n) {
        if (n < 100) return convertLessThanHundred(n);
        return ones[Math.floor(n / 100)] + ' Hundred' + (n % 100 ? ' ' + convertLessThanHundred(n % 100) : '');
    }

    if (num < 0) return 'Negative ' + numberToWords(-num);
    if (num < 1000) return convertLessThanThousand(num);
    if (num < 100000) {
        return convertLessThanHundred(Math.floor(num / 1000)) + ' Thousand' +
               (num % 1000 ? ' ' + convertLessThanThousand(num % 1000) : '');
    }
    if (num < 10000000) {
        return convertLessThanHundred(Math.floor(num / 100000)) + ' Lakh' +
               (num % 100000 ? ' ' + numberToWords(num % 100000) : '');
    }
    return convertLessThanHundred(Math.floor(num / 10000000)) + ' Crore' +
           (num % 10000000 ? ' ' + numberToWords(num % 10000000) : '');
}
</script>
@endsection
